// ------------------------------------------------------------------------
// audiofx_amplitude.cpp: Amplitude effects and dynamic processors.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <cmath>

#include <kvutils.h>

#include "samplebuffer.h"
#include "audiofx_amplitude.h"

#include "eca-debug.h"
#include "eca-error.h"

EFFECT_AMPLIFY::EFFECT_AMPLIFY (parameter_type multiplier_percent =
				100.0) {
  set_parameter(1, multiplier_percent);
}

void EFFECT_AMPLIFY::set_parameter(int param, parameter_type value) {
  switch (param) {
  case 1: 
    kerroin = value / 100.0;
    break;
  }
}

DYNAMIC_PARAMETERS::parameter_type EFFECT_AMPLIFY::get_parameter(int param) const { 
  switch (param) {
  case 1: 
    return(kerroin * 100.0);
  }
  return(0.0);
}

void EFFECT_AMPLIFY::process(SAMPLE_BUFFER* sbuf) {
  sbuf->first();
  while(sbuf->is_readable()) {
    sbuf->current_sample()->multiply(kerroin);
    sbuf->next();
  }
}

EFFECT_AMPLIFY_CLIPCOUNT::EFFECT_AMPLIFY_CLIPCOUNT (parameter_type multiplier_percent, int max_clipped) {

  set_parameter(1, multiplier_percent);
  set_parameter(2, max_clipped);
  num_of_clipped = 0;
}

void EFFECT_AMPLIFY_CLIPCOUNT::set_parameter(int param, parameter_type value) {
  switch (param) {
  case 1: 
    kerroin = value / 100.0;
    break;
  case 2:
    maxnum_of_clipped = (int)value;
    break;
  }
}

DYNAMIC_PARAMETERS::parameter_type EFFECT_AMPLIFY_CLIPCOUNT::get_parameter(int param) const { 
  switch (param) {
  case 1: 
    return(kerroin * 100.0);
  case 2:
    return(maxnum_of_clipped);
  }
  return(0.0);
}

void EFFECT_AMPLIFY_CLIPCOUNT::process(SAMPLE_BUFFER* sbuf) {
  sbuf->first();
  while(sbuf->is_readable()) {
    sbuf->current_sample()->multiply(kerroin);
    if (sbuf->current_sample()->is_clipped() == true) {
      num_of_clipped++;
    }
    else {
      num_of_clipped = 0;
    }
    sbuf->next();
  }

  if (num_of_clipped > maxnum_of_clipped && maxnum_of_clipped != 0) {
    MESSAGE_ITEM otemp;
    otemp.setprecision(0);
    otemp << "(audiofx_amplitude) WARNING! Signal is clipping! ";
    otemp << num_of_clipped;
    otemp << " consecutive clipped samples.";
    ecadebug->msg(otemp.to_string());
  }
}

EFFECT_AMPLIFY_CHANNEL::EFFECT_AMPLIFY_CHANNEL (parameter_type multiplier_percent, int channel) {
  set_parameter(1, multiplier_percent);
  set_parameter(2, channel);
}

void EFFECT_AMPLIFY_CHANNEL::set_parameter(int param, parameter_type value) {
  switch (param) {
  case 1: 
    kerroin = value / 100.0;
    break;

 case 2: 
    channel_rep = static_cast<int>(value);
    break;
  }
}

DYNAMIC_PARAMETERS::parameter_type EFFECT_AMPLIFY_CHANNEL::get_parameter(int param) const { 
  switch (param) {
  case 1: 
    return(kerroin * 100.0);

  case 2: 
    return(static_cast<parameter_type>(channel_rep));
  }
  return(0.0);
}

void EFFECT_AMPLIFY_CHANNEL::process(SAMPLE_BUFFER::sample_type *insample) {
    insample->put_channel(channel_rep,
			  insample->get_channel(channel_rep) 
			  * kerroin);
}

EFFECT_COMPRESS::EFFECT_COMPRESS (parameter_type compress_rate, parameter_type thold) {
  // map_parameters();

  set_parameter(1, compress_rate);
  set_parameter(2, thold);

  first_time = true;
}

EFFECT_COMPRESS::EFFECT_COMPRESS (const EFFECT_COMPRESS& x) {
  crate = x.crate;
  threshold = x.threshold;
  delta = x.delta;
  first_time = x.first_time;
  for(nm = 0;nm < SAMPLE_BUFFER::ch_count; nm++)
    s[nm] = x.s[nm];
  temp = x.temp;
  lastin = x.lastin;
  lastout = x.lastout;
}

void EFFECT_COMPRESS::set_parameter(int param, parameter_type value) {
  switch (param) {
  case 1: 
    crate = pow(2.0, value / 3.0);
    break;
  case 2: 
    threshold = value;
    break;
  }
}

DYNAMIC_PARAMETERS::parameter_type EFFECT_COMPRESS::get_parameter(int param) const { 
  switch (param) {
  case 1: 
    return((log (crate)) / (log (2)) * 3.0);
  case 2: 
    return(threshold);
  }
  return(0.0);
}

void EFFECT_COMPRESS::process(SAMPLE_BUFFER::sample_type *insample) {
  if (first_time) {
    first_time = false;
    lastin = lastout = *insample;
  }
  else {
    for(nm = 0; nm < SAMPLE_BUFFER::ch_count; nm++) {
      delta = insample->get_channel(nm) / lastin.get_channel(nm);

      if (fabs(insample->get_channel(nm)) > threshold) {
	if (delta > 1.0) {
	  delta = (delta - 1.0) / crate;
	  delta += 1.0;
	}
	new_value = lastout.get_channel(nm) * delta;
      }
      else
	new_value = lastin.get_channel(nm) * delta;

      if (new_value > SAMPLE_SPECS::max_value) new_value = SAMPLE_SPECS::max_value;
      else if (new_value < SAMPLE_SPECS::min_value) new_value = SAMPLE_SPECS::min_value;
      temp.put_channel(nm, new_value);
    }
    lastin = *insample;
    lastout = temp;
    *insample = temp;
  }
}

EFFECT_NOISEGATE_MONO::EFFECT_NOISEGATE_MONO (parameter_type thlevel_percent, parameter_type thtime, parameter_type a, parameter_type h, parameter_type r) {
  // map_parameters();

  set_parameter(1, th_level);
  set_parameter(2, th_time);
  set_parameter(3, atime);
  set_parameter(4, htime);
  set_parameter(5, rtime);
  
  ng_status = ng_waiting;
}

void EFFECT_NOISEGATE_MONO::set_parameter(int param, parameter_type value) {
  switch (param) {
  case 1: 
    th_level = SAMPLE_BUFFER::max_amplitude * (value / 100.0);
    th_time_lask = 0.0;
    break;
  case 2: 
    th_time = (value * (parameter_type)SAMPLE_BUFFER::sample_rate * (parameter_type)SAMPLE_BUFFER::ch_count / 1000.0);
    break;
  case 3: 
    atime = (value * (parameter_type)SAMPLE_BUFFER::sample_rate * (parameter_type)SAMPLE_BUFFER::ch_count / 1000.0);
    attack_lask = 0.0;
    break;
  case 4: 
    htime = (value * (parameter_type)SAMPLE_BUFFER::sample_rate * (parameter_type)SAMPLE_BUFFER::ch_count / 1000.0);
    hold_lask = 0.0;
    break;
  case 5: 
    rtime = (value * (parameter_type)SAMPLE_BUFFER::sample_rate * (parameter_type)SAMPLE_BUFFER::ch_count / 1000.0);
    release_lask = 0.0;
    break;
  }
}

DYNAMIC_PARAMETERS::parameter_type EFFECT_NOISEGATE_MONO::get_parameter(int param) const { 
  switch (param) {
  case 1: 
    return(th_level * 100.0 / (parameter_type)SAMPLE_BUFFER::max_amplitude);
  case 2: 
    return(th_time * 1000.0 / (parameter_type)SAMPLE_BUFFER::sample_rate / (parameter_type)SAMPLE_BUFFER::ch_count);
  case 3: 
    return(atime * 1000.0 / (parameter_type)SAMPLE_BUFFER::sample_rate / (parameter_type)SAMPLE_BUFFER::ch_count);
  case 4: 
    return(htime * 1000.0 / (parameter_type)SAMPLE_BUFFER::sample_rate / (parameter_type)SAMPLE_BUFFER::ch_count);
  case 5: 
    return(rtime * 1000.0 / (parameter_type)SAMPLE_BUFFER::sample_rate / (parameter_type)SAMPLE_BUFFER::ch_count);
  }
  return(0.0);
}

void EFFECT_NOISEGATE_MONO::process(SAMPLE_BUFFER::sample_type *insample) {
  temp_sample = *insample;
  temp_sample.sum_to_mono();
    
  if (fabs((parameter_type)insample->get_left()) <= th_level) {
    if (ng_status == ng_holding || ng_status == ng_releasing) {
      ng_status = ng_waiting; 
      ecadebug->msg(10,"(audiofx) ng_mono waiting (again)");
    }

    if (ng_status == ng_active) {
      insample->make_silent();
      return;
    }

    if (ng_status == ng_attacking) {
      attack_lask++;
      kerroin = (1 - (attack_lask / atime));
      if (attack_lask >= atime) {
	ng_status = ng_active;
	ecadebug->msg(10,"(audiofx) ng_mono active");
	attack_lask = 0.0;
      }
    }

    if (ng_status == ng_waiting) {
      th_time_lask++;
      if (th_time_lask >= th_time) {
	ng_status = ng_attacking;
	ecadebug->msg(10,"(audiofx) ng_mono attacking");
	th_time_lask = 0.0;
      }
      return;
    }
  }
  else {
    if (ng_status == ng_releasing) {
      release_lask++;
      kerroin = release_lask / rtime;
      if (release_lask >= rtime) {
	ecadebug->msg(10,"(audiofx) ng_mono waiting");
        ng_status = ng_waiting;
	release_lask = 0.0;
      }
    }

    if (ng_status == ng_holding) {
      hold_lask++;
      if (hold_lask >= htime) {
	ng_status = ng_releasing;
	ecadebug->msg(10,"(audiofx) ng_mono releasing");
	hold_lask = 0.0;
      }
      insample->make_silent();
      return;
    }

    if (ng_status == ng_active) {
      ng_status = ng_holding;
      ecadebug->msg(10,"(audiofx) ng_mono holding");
      return;
    }

    if (ng_status == ng_waiting
	|| ng_status == ng_attacking) {
      ng_status = ng_waiting; 
      th_time_lask = 0.0;
      //      ecadebug->msg(10,"(audiofx) ng_mono waiting (reseted)");
      return;
    }
  }

  insample->put_left(insample->get_left() * kerroin);
  insample->put_right(insample->get_right() * kerroin);
}

EFFECT_NORMAL_PAN::EFFECT_NORMAL_PAN (parameter_type right_percent) {
  // map_parameters();
  set_parameter(1, right_percent);
}

void EFFECT_NORMAL_PAN::set_parameter(int param, parameter_type value) {
  switch (param) {
  case 1: 
    right_percent_rep = value;
    if (value == 50.0) {
      l_kerroin = r_kerroin = 1.0;
    }
    else if (value < 50.0) {
      l_kerroin = 1.0;
      r_kerroin = value / 50.0;
    }
    else if (value > 50.0) {
      r_kerroin = 1.0;
      l_kerroin = (100.0 - value) / 50.0;
    }
    break;
  }
}

DYNAMIC_PARAMETERS::parameter_type EFFECT_NORMAL_PAN::get_parameter(int param) const { 
  switch (param) {
  case 1: 
    return(right_percent_rep);
  }
  return(0.0);
}

void EFFECT_NORMAL_PAN::process(SAMPLE_BUFFER::sample_type *insample) {
  insample->put_left(insample->get_left() * l_kerroin);
  insample->put_right(insample->get_right() * r_kerroin);
}
