#ifndef _AUDIOFX_TIMEBASED_H
#define _AUDIOFX_TIMEBASED_H

#include <vector>
#include <string>

#include "audiofx.h"

typedef deque<SAMPLE_BUFFER::sample_type> SINGLE_BUFFER;

/**
 * Base class for time-based effects (delays, reverbs, etc).
 */
class EFFECT_TIME_BASED : public EFFECT_BASE {

 protected:
    
  parameter_type dtime;        // delay time in samples
  parameter_type laskuri;
  
 public:

  EFFECT_TIME_BASED (void) : dtime(0.0), laskuri(0.0) { }
  virtual EFFECT_TIME_BASED* clone(void) = 0;
};

/** 
 * Delay effect.
 */
class EFFECT_DELAY : public EFFECT_TIME_BASED {

 private:

  SAMPLE_BUFFER::sample_type temp, temp2;     // temporary values
  int nm2; // temp-value

  parameter_type surround;
  parameter_type dnum;
  parameter_type mix;

  vector<SINGLE_BUFFER> buffer;

 public:

  string name(void) { return("Delay"); }

  vector<string> parameter_names(void) const {
    vector<string> t;
    t.push_back("delay_time");
    t.push_back("surround_mode");
    t.push_back("number_of_delays");
    t.push_back("mix_%");
    return(t);
  }

  parameter_type get_parameter(int param) const;
  void set_parameter(int param, parameter_type value);

  void process(SAMPLE_BUFFER::sample_type *insample);
  parameter_type get_delta_in_samples(void) { return(dnum * dtime); }

  EFFECT_DELAY* clone(void)  { return new EFFECT_DELAY(*this); }
  EFFECT_DELAY (parameter_type delay_time = 0.0, int surround_mode = 0, int num_of_delays = 1, parameter_type mix_percent = 50.0);
};

/*
 * Transforms a mono signal to stereo using a panned delay signal.
 * Suitable delays values range from 1 to 40 milliseconds. 
 */
class EFFECT_FAKE_STEREO : public EFFECT_TIME_BASED {

  SAMPLE_BUFFER::sample_type temp;   // temp-values

  deque<SAMPLE_BUFFER::sample_type> buffer;

 public:

  string name(void) { return("Fake stereo"); }

  vector<string> parameter_names(void) const {
    vector<string> t;
    t.push_back("delay_time");
    return(t);
  }

  parameter_type get_parameter(int param) const;
  void set_parameter(int param, parameter_type value);

  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_FAKE_STEREO* clone(void)  { return new EFFECT_FAKE_STEREO(*this); }
  EFFECT_FAKE_STEREO::EFFECT_FAKE_STEREO (parameter_type delay_time = 0.0);
};

/*
 * Simple reverb effect.
 */
class EFFECT_REVERB : public EFFECT_TIME_BASED {

 private:
    
  SAMPLE_BUFFER::sample_type temp;  // temp-values

  deque<SAMPLE_BUFFER::sample_type> buffer;

  parameter_type surround;
  parameter_type feedback;

 public:

  string name(void) { return("Reverb"); }

  vector<string> parameter_names(void) const {
    vector<string> t;
    t.push_back("delay_time");
    t.push_back("surround_mode");
    t.push_back("feedback_%");
    return(t);
  }

  parameter_type get_parameter(int param) const;
  void set_parameter(int param, parameter_type value);

  void process(SAMPLE_BUFFER::sample_type *insample);
  parameter_type get_delta_in_samples(void) { return(dtime); }

  EFFECT_REVERB* clone(void)  { return new EFFECT_REVERB(*this); }
  EFFECT_REVERB (parameter_type delay_time = 0.0, int surround_mode = 0, parameter_type feedback_percent = 50.0);
};

#endif


