// ------------------------------------------------------------------------
// audioio-af.cpp: Interface to SGI audiofile library.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <config.h>
#ifdef COMPILE_AF

#include <algorithm>
#include <string>
#include <iostream.h>
#include <fstream.h>
#include <cmath>
#include <cstring>
#include <audiofile.h>

#include <kvutils.h>

#include "audioio-af.h"
#include "samplebuffer.h"

#include "eca-error.h"
#include "eca-debug.h"
  
AUDIOFILE_INTERFACE::AUDIOFILE_INTERFACE (const string& name, const
					  SIMODE mode, const
					  AIO_PARAMS& fmt, int bsize) 
  :  AUDIO_IO_FILE(name, mode, fmt, bsize) {

  // --
  // Initialize attributes.
  // --
  position_in_samples(0);
}

void AUDIOFILE_INTERFACE::open_device(void) {

  switch(io_mode()) {
  case si_read:
    {
      ecadebug->msg("(audioio-af) Using audiofile library to open file \"" +
		    label() + "\" for reading.");

      afhandle = afOpenFile(label().c_str(), "r", NULL);
      if (afhandle == AF_NULL_FILEHANDLE) {
	throw(new ECA_ERROR("AUDIOIO-AF", "Can't open file \"" + label()
			    + "\" using libaudiofile."));
      }
      AIO_PARAMS aiotemp;
      int sample_format, sample_width;
      
      aiotemp.srate = (long int)afGetRate(afhandle, AF_DEFAULT_TRACK);
      aiotemp.channels = afGetChannels(afhandle, AF_DEFAULT_TRACK);
      afGetSampleFormat(afhandle, AF_DEFAULT_TRACK, &sample_format, &sample_width);
      aiotemp.bits = sample_width;
      format(aiotemp);
      
      length_in_samples(afGetFrameCount(afhandle, AF_DEFAULT_TRACK));
      break;
    }
  case si_write:
    {
      ecadebug->msg("(audioio-af) Using audiofile library to open file \"" +
		    label() + "\" for writing.");

      AFfilesetup fsetup;
      fsetup = afNewFileSetup();

      int file_format = -1;
      string teksti = label();
      to_lowercase(teksti);
      if (strstr(teksti.c_str(),".aif") != 0) {
	file_format = AF_FILE_AIFF;
      }
      else if (strstr(teksti.c_str(),".au") != 0) {
	file_format = AF_FILE_NEXTSND;
      }
      else if (strstr(teksti.c_str(),".snd") != 0) {
	file_format = AF_FILE_NEXTSND;
      }
      else {
	ecadebug->msg("(audioio-af) Warning! Unknown audio format, using raw format instead.");
	file_format = AF_FILE_RAWDATA;
      }
      afInitFileFormat(fsetup, file_format);
      afInitChannels(fsetup, AF_DEFAULT_TRACK, format().channels);
      if (format().bits == 8) 
	afInitSampleFormat(fsetup, AF_DEFAULT_TRACK, AF_SAMPFMT_UNSIGNED, 8);
      else 
	afInitSampleFormat(fsetup, AF_DEFAULT_TRACK, AF_SAMPFMT_TWOSCOMP, format().bits);
      afInitRate(fsetup, AF_DEFAULT_TRACK, (double)format().srate);

      afhandle = afOpenFile(label().c_str(), "w", fsetup);
      if (afhandle == AF_NULL_FILEHANDLE) 
	throw(new ECA_ERROR("AUDIOIO-AF", "Can't open file \"" + label()
			    + "\" using libaudiofile."));
     break;
    }
  
  case si_readwrite:
    {
      throw(new ECA_ERROR("AUDIOIO-AF", "Simultaneous intput/ouput not supported."));
    }
  }

  // --
  // Get byteorder
  // --
  //  if (SAMPLE_BUFFER::is_system_littleendian)
  //    afSetVirtualByteOrder(afhandle, AF_DEFAULT_TRACK, AF_BYTEORDER_LITTLEENDIAN);
  //  else
  //    afSetVirtualByteOrder(afhandle, AF_DEFAULT_TRACK, AF_BYTEORDER_BIGENDIAN);

  debug_print_type();
  set_open_state(true);
  position_in_samples(position_in_samples(), true);
}

void AUDIOFILE_INTERFACE::close_device(void) {
  if (is_open()) {
    afCloseFile(afhandle);
    set_open_state(false);
  }
}

AUDIOFILE_INTERFACE::~AUDIOFILE_INTERFACE(void) {
  close_device();
}

void AUDIOFILE_INTERFACE::debug_print_type(void) {
  int temp = afGetFileFormat(afhandle, 0);
  ecadebug->msg(1, "(audioio-af) afFileformat: " + kvu_numtostr(temp)
		+ "."); 
}

void AUDIOFILE_INTERFACE::read_buffer(SAMPLE_BUFFER* t) {
  //  curpos_in_samples += t->length_in_samples();
  t->reserve_buffer(buffersize());

  samples_read = afReadFrames(afhandle, AF_DEFAULT_TRACK, (void*)(t->iobuf_uchar), buffersize());
  t->iobuf_to_buffer(samples_read,
		     SAMPLE_BUFFER::is_system_littleendian,
		     format().bits, format().channels, format().srate);

  position_in_samples(position_in_samples() + t->length_in_samples());
  if (position_in_samples() >= length_in_samples()) finished(true);
}

void AUDIOFILE_INTERFACE::write_buffer(SAMPLE_BUFFER* t) {
  //    curpos_in_samples += t->length_in_samples();
  t->buffer_to_iobuf(SAMPLE_BUFFER::is_system_littleendian, format().bits, format().channels, format().srate);
  position_in_samples(position_in_samples() + t->length_in_samples());
  length_in_samples(position_in_samples());
  afWriteFrames(afhandle, AF_DEFAULT_TRACK, (void*)(t->iobuf_uchar),
		t->length_in_samples());
}

void AUDIOFILE_INTERFACE::position_in_samples_hook(void) {
  afSeekFrame(afhandle, AF_DEFAULT_TRACK, position_in_samples());
  finished(false);
}

#endif
