#ifndef _ECA_CHAINSETUP_H
#define _ECA_CHAINSETUP_H

#include <vector>
#include <string>
#include <map>

#include <kvutils.h>

#include "eca-chain.h"
#include "eca-region.h"
#include "audioio.h"

class CONTROLLER_SOURCE;

class CHAIN_OPERATOR;
class ECA_RESOURCES;

/**
 * Class that represents a single chainsetup.
 * @author Kai Vehmanen
 */
class ECA_CHAINSETUP : public ECA_REGION {

 public:

  enum EP_MM_MODE { ep_mm_auto, ep_mm_simple, ep_mm_normal, ep_mm_mthreaded };

  enum {
    TYPE_NOT_OPENED =   -1,
    TYPE_UNKNOWN =       0,
    TYPE_WAVE =          1,
    TYPE_DART =          2,
    TYPE_CDR =           3,
    TYPE_OSS =           4,
    TYPE_EWF =           5,
    TYPE_ESS =           6,
    TYPE_OSSDMA =        7,
    TYPE_MP3 =           8,
    TYPE_ALSA =          9,
    TYPE_ALSAFILE =     10,
    TYPE_ALSALOOPBACK = 16,
    TYPE_AUDIOFILE =    11,
    TYPE_RAWFILE =      12,
    TYPE_STDIN =        13,
    TYPE_STDOUT =       14,
    TYPE_NULL =         15
  };

 public:

  static map<string, CHAIN_OPERATOR*> chainop_map;
  static map<string, string> chainop_prefix_map;

 private:

  // ---
  // Setup data
  // ---
  vector<string> active_chainids;
  int chaincount;
  AIO_PARAMS active_sinfo;
  bool active_sinfo_explicit;

  map<CHAIN::aio_id_type,vector<string> > chains_assigned_to_idev;
  map<CHAIN::aio_id_type,vector<string> > chains_assigned_to_odev;
  map<CHAIN::aio_id_type,int> number_of_chains_assigned_to_idev;
  map<CHAIN::aio_id_type,int> number_of_chains_assigned_to_odev;

  string setup_name;
  string setup_filename;

  bool active;
  ECA_RESOURCES* ecaresources;

  vector<string> options;
  string options_general, options_inputs, options_outputs, options_chains;

  // ---
  // Setup to strings
  // ---
  void update_option_strings(void);
  string general_options_to_string(void);
  string inputs_to_string(void);
  string outputs_to_string(void);
  string chains_to_string(void);
  string audioio_to_string(AUDIO_IO* aiod, const string& direction);
  string chainop_to_string(CHAIN_OPERATOR* chainop, vector<GCONTROLLER*>& gctrls);
  string gcontroller_to_string(GCONTROLLER* gctrl);

  // ---
  // Setup helper functions
  // ---
  AUDIO_IO* new_aio_device(const string& tname, const SIMODE mode, const AIO_PARAMS& format, int buffersize);

  /**
   * Adds a "default" chain to the setup.
   *
   * require:
   *   buffersize >= 0 && chains.size() == 0
   *
   * ensure:
   *   chains.back()->name() == "default" && 
   *   active_chainids.back() == "default"
   */
  void add_default_chain(void);

  /**
   * Update how inputs and outputs are mapped to chains.
   * 
   */
  void update_chain_mappings(void);

  /**
   * Recognise file format from the file name.
   */
  int get_type_from_extension (const string& teksti);

  /**
   * Get the prefix part of a string argument
   * @param argument format used is -prefix:arg1, arg2, ..., argN
   *
   * require:
   *   argu.find('-') != string::npos
   *
   * ensure:
   *   argu.size() >= 0
   */
  string get_argument_prefix(const string& argument);

  /**
   * Print format and id information
   *
   * require:
   *   aio != 0
   */
  string open_info(const AUDIO_IO* aio);

 public:

  /**
   * Register a new effect.
   */
  static void register_chain_operator(const string& id_string,
				      CHAIN_OPERATOR* chainops);

  /**
   * Register default effects.
   */
  static void register_default_chainops(void);

  /**
   * Set default values.
   */
  void set_defaults (void);

  void set_name(const string& str) { setup_name = str; }
  void set_filename(const string& str) { setup_filename = str; }

  /**
   * Handle options. 
   */
  void interpret_options(void);

  /**
   * Handle general options. 
   */
  void interpret_general_option (const string& argu);

  /**
   * Handle region options.
   */
  void interpret_region (const string& argu);

  /**
   * Handle mixmode options.
   */
  void interpret_mix_mode (const string& argu);

  /**
   * Handle chainsetup options.
   */
  void interpret_setup_state (const string& argu);

  /**
   * Handle audio-IO-devices and files.
   */
  void interpret_audioio_device (const string& argu, const string& argu_param);

  /**
   * Handle chain operator options.
   */
  void interpret_chain_operator (const string& argu);

  /**
   * Handle controller sources and general controllers.
   */
  void interpret_controller (const string& argu);

  /**
   * Handle effect preset options.
   */
  void interpret_effect_preset (const string& argu);

  /**
   * Process a singlechain effect preset.
   */
  void add_singlechain_preset(const string& preset_name);

  // ---
  // Setup functions (except chains themselves, all additions are done
  // to all active chains)
  // ---

  /**
   * Enable chainsetup. Opens all devices and reinitializes all 
   * chain operators if necessary.
   * 
   * ensure:
   *   is_enabled() == false
   */
  void enable(void);

  /**
   * Disable chainsetup. Closes all devices. 
   * 
   * ensure:
   *   is_enabled() == false
   */
  void disable(void);

  /**
   * Checks whethers chainsetup is enabled (device ready for use).
   */
  bool is_enabled(void) { return(active); }

  void add_new_chains(const vector<string>& newchains);
  void remove_chain(const string& name);
  void set_active_chains(const vector<string>& chains) { active_chainids = chains; }
  void set_explicit_format(bool enabled) { active_sinfo_explicit = enabled; }

  /**
   * Add chain operator to active chains. If no chains exist, 
   * a default chain is created. 
   *
   * require:
   *   cotmp != 0
   *
   * ensure:
   *   chains.size() != 0
   */
  void add_chainop(CHAIN_OPERATOR* cotmp);

  /**
   * Add general controller to active chains. 
   *
   * require:
   *   csrc != 0 && fxparam > 0
   */
  void add_gcontroller(CONTROLLER_SOURCE* csrc, int fxparam, double low, double high);

  /**
   * Add a new input object to active chains.
   *
   * require:
   *   aiod != 0 && chains.size() > 0
   *
   * ensure:
   *   inputs.size() > 0
   */
  void add_input(AUDIO_IO* aiod);

  /**
   * Add a new output object to active chains.
   *
   * require:
   *   aiod != 0 && chains.size() > 0
   *
   * ensure:
   *   outputs.size() > 0
   */
  void add_output(AUDIO_IO* aiod);
  void remove_audio_device(const string& label);
  void attach_iodev_to_active_chains(const string& filename);

  // ---
  // Setup load/save functions
  // ---
  void load_from_file(const string& filename);
  void save(void);
  void save_to_file(const string& filename);

  // ---
  // Status/info functions
  // ---
  vector<string> get_connected_chains_to_input(CHAIN::aio_id_type
					       aiod) { return(chains_assigned_to_idev[aiod]); }
  vector<string> get_connected_chains_to_output(CHAIN::aio_id_type
						aiod) { return(chains_assigned_to_odev[aiod]); }
  vector<string> get_connected_chains_to_iodev(const string& filename);
  int number_of_connected_chains_to_input(CHAIN::aio_id_type aiod) {
    return(number_of_chains_assigned_to_idev[aiod]); }
  int number_of_connected_chains_to_output(CHAIN::aio_id_type aiod) {
    return(number_of_chains_assigned_to_odev[aiod]); }
  const CHAIN* get_chain_with_name(const string& name) const;

  // ---
  // Public data objects
  // ---
  vector<AUDIO_IO*> inputs;
  vector<AUDIO_IO*> outputs;
  vector<CHAIN*> chains;
  int buffersize;        // Buffersize in samples.
  long int sample_rate;  // sample rate (Hz)
  bool raisepriority;    // Should engine raise its process priority?
  bool double_buffering; // Whether to use double-buffering on audio
                         // inputs that support it.
  enum EP_MM_MODE mixmode;
  SIMODE output_openmode;

  // ---
  // Chainsetup info
  // --
  const string& name(void) const { return(setup_name); }
  const string& filename(void) const { return(setup_filename); }
  bool is_valid(void) const;

  /**
   * Construct from a command line object.
   *
   * require:
   *   ecarc != 0
   *
   * ensure:
   *   buffersize != 0
   */
  ECA_CHAINSETUP(ECA_RESOURCES* ecarc, COMMAND_LINE& cline);

  /**
   * Construct from a chainsetup file.
   *
   * require:
   *   ecarc != 0
   *
   * ensure:
   *   buffersize != 0
   */
  ECA_CHAINSETUP(ECA_RESOURCES* ecarc, const string& setup_file, bool fromfile = true);
    
  /**
   * Destructor
   */
  ~ECA_CHAINSETUP(void);
};

#endif


