// ------------------------------------------------------------------------
// audioio-alsalb.cpp: ALSA (/dev/snd/pcm*) loopback input.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <config.h>
#ifdef COMPILE_ALSA

#include <string>
#include <cstring>
#include <cstdio>
#include <dlfcn.h>  

#include <kvutils.h>

#include "samplebuffer.h"
#include "audioio.h"
#include "audioio-alsalb.h"
#include "eca-alsa-dyn.h"

#include "eca-error.h"
#include "eca-debug.h"

ALSALBDEVICE::ALSALBDEVICE (int card, int device, const SIMODE
			mode, const AIO_PARAMS& form, int bsize,
			bool playback_mode)
  :  AUDIO_IO_DEVICE(string("alsa_loopback,") + kvu_numtostr(card) +
		     string(",") + kvu_numtostr(device) , mode, form, bsize)
{
  card_number = card;
  device_number = device;
  finished(false);
  set_open_state(false);
  is_triggered = false;

  pb_mode = playback_mode;

  eca_alsa_load_dynamic_support();
}

void ALSALBDEVICE::open_device(void) {
  if (is_open() == true) return;
  int err;
  if (io_mode() == si_read) {
    if (pb_mode) {
      if ((err = dl_snd_pcm_loopback_open(&audio_fd, card_number, device_number,
					  SND_PCM_LB_OPEN_PLAYBACK)) < 0) {
	throw(new ECA_ERROR("AUDIOIO-ALSALB", "unable to open ALSA-device for reading; error: " + string(dl_snd_strerror(err))));
      }
    }
    else {
      if ((err = dl_snd_pcm_loopback_open(&audio_fd, card_number, device_number,
					  SND_PCM_LB_OPEN_CAPTURE)) < 0) {
	throw(new ECA_ERROR("AUDIOIO-ALSALB", "unable to open ALSA-device for reading; error: " + string(dl_snd_strerror(err))));
      }
    }
  }    
  else {
      throw(new ECA_ERROR("AUDIOIO-ALSALB", "Only readinng support with a loopback device."));
  }
  
  // ---
  // Set blocking mode.
  // ---
  dl_snd_pcm_loopback_block_mode(audio_fd, 1);    // enable block mode

  // ---
  // Set fragment size.
  // ---
  if (buffersize() == 0) 
    throw(new ECA_ERROR("AUDIOIO-ALSALB", "Buffersize() is 0!", ECA_ERROR::stop));
  
  // ---
  // Select audio format
  // ---

  snd_pcm_format_t pf;
  memset(&pf, 0, sizeof(pf));
  if (format().bits == 16)
    pf.format = SND_PCM_SFMT_S16_LE; // DL_SND_PCM_FMT_U16_LE
  else 
    pf.format = SND_PCM_SFMT_U8;

  pf.rate = format().srate;
  pf.channels = format().channels; // Stereo

  if (io_mode() == si_read) {
    err = dl_snd_pcm_loopback_format(audio_fd, &pf);
    if (err < 0) {
    throw(new ECA_ERROR("AUDIOIO-ALSALB", "Error when setting up record parameters: " + string(dl_snd_strerror(err))));
    }
    //    AIO_PARAMS temp;
    //    if (pf.format = SND_PCM_SFMT_S16_LE)
    //      temp.bits = 16;
    //    else 
    //      temp.bits = 8;
    //
    //    temp.srate = pf.rate;
    //    temp.channels = pf.channels;
    //    format(temp);
  }
  set_open_state(true);
}

void ALSALBDEVICE::close_device(void) {
  if (is_open()) {
    dl_snd_pcm_loopback_close(audio_fd);
  }    
  set_open_state(false);
}

void ALSALBDEVICE::read_buffer(SAMPLE_BUFFER* t) {
  t->reserve_buffer(buffersize());
  if (!is_open()) throw(new ECA_ERROR("AUDIOIO-ALSALB","get_sample(): trying to read from a closed device!"));

  bytes_read = dl_snd_pcm_loopback_read(audio_fd, t->iobuf_uchar, format().align * buffersize());

  if (bytes_read < 0)
    throw(new ECA_ERROR("AUDIOIO-ALSALB","get_sample(): read error! - " + string((*dl_snd_strerror)(bytes_read)), ECA_ERROR::stop));

  //    t->length_in_samples(samples_read / format().align);
  //  else 

  t->iobuf_to_buffer(bytes_read / format().align, true, format().bits, format().channels, format().srate);
}

ALSALBDEVICE::~ALSALBDEVICE(void) { 
  close_device();

  eca_alsa_unload_dynamic_support();
}

#endif // COMPILE_ALSA

