// ------------------------------------------------------------------------
// audiofx_filter.cpp: Routines for filter effects.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <cmath>

#include <kvutils.h>

#include "samplebuffer.h"
#include "eca-debug.h"
#include "audiofx_filter.h"

EFFECT_BANDPASS::EFFECT_BANDPASS (double centerf, double w) {
  add_parameter("center-freq");
  add_parameter("width");

  set_parameter(1, centerf);
  set_parameter(2, w);

  MESSAGE_ITEM otemp;
  otemp.setprecision(2);
  otemp << "FX:    Bandpass filter enabled, center frequency is ";
  otemp << center << " and width " << width << ".";
  ecadebug->msg(otemp.to_string());
}

void EFFECT_BANDPASS::set_parameter(int param, double value) {
  switch (param) {
  case 1: 
    center = value;
    D = 2 * cos(2 * M_PI * center / (double)SAMPLE_BUFFER::sample_rate);
    constant_b(0, -C * D * constant_a(0));
    break;
  case 2: 
    if (value != 0) width = value;
    else width = center / 2;
    C = 1.0 / tan(M_PI * width / (double)SAMPLE_BUFFER::sample_rate);
    D = 2 * cos(2 * M_PI * center / (double)SAMPLE_BUFFER::sample_rate);
    constant_a(0,1.0 / (1.0 + C));
    constant_a(1,0.0);
    constant_a(2,-constant_a(0));
    constant_b(1, (C - 1.0) * constant_a(0));
    break;
  }
}

double EFFECT_BANDPASS::get_parameter(int param) const { 
  switch (param) {
  case 1: 
    return(center);
  case 2: 
    return(width);
  }
  return(0.0);
}

EFFECT_BANDREJECT::EFFECT_BANDREJECT (double centerf, double w) 
{
  add_parameter("center-freq");
  add_parameter("width");

  set_parameter(1, centerf);
  set_parameter(2, w);
  
  MESSAGE_ITEM otemp;
  otemp.setprecision(2);
  otemp << "FX:    Bandreject filter enabled, center frequency is ";
  otemp << center << " and width " << width << ".";
  ecadebug->msg(otemp.to_string());
}

void EFFECT_BANDREJECT::set_parameter(int param, double value) {
  switch (param) {
  case 1: 
    center = value;
    D = 2 * cos(2 * M_PI * center / (double)SAMPLE_BUFFER::sample_rate);
    constant_a(1, -D * constant_a(0));
    constant_b(0, constant_a(1));
    break;
  case 2: 
    if (value != 0) width = value;
    else width = center / 2;
    C = tan(M_PI * width / (double)SAMPLE_BUFFER::sample_rate);
    constant_a(0, 1.0 / (1.0 + C));
    constant_a(1, -D * constant_a(0));
    constant_a(2, constant_a(0));
    constant_b(0, constant_a(1));
    constant_b(1, (1.0 - C) * constant_a(0));
    break;
  }
}

double EFFECT_BANDREJECT::get_parameter(int param) const { 
  switch (param) {
  case 1: 
    return(center);
  case 2: 
    return(width);
  }
  return(0.0);
}

void EFFECT_BW_FILTER::process(SAMPLE_BUFFER::sample_type *insample) {
  for(nm = 0; nm < SAMPLE_BUFFER::ch_count; nm++) {
    outputSample.put_channel(nm, a[0] * insample->get_channel(nm) + a[1]
			     * sin[0].get_channel(nm) + a[2] * sin[1].get_channel(nm)
			     - b[0] * sout[0].get_channel(nm) - b[1] * sout[1].get_channel(nm));
    sin[1].put_channel(nm, sin[0].get_channel(nm));
    sin[0].put_channel(nm, insample->get_channel(nm));
   
    sout[1].put_channel(nm, sout[0].get_channel(nm));
    sout[0].put_channel(nm, outputSample.get_channel(nm));
  }
  insample->operator=(outputSample);
}

void EFFECT_BW_FILTER::init_values(void) {
  for(int j = 0; j < 2;j++) {
    sin[j].sample[SAMPLE_BUFFER::ch_left] = 0.0;
    sin[j].sample[SAMPLE_BUFFER::ch_right] = 0.0;
    sout[j].sample[SAMPLE_BUFFER::ch_left] = 0.0;
    sout[j].sample[SAMPLE_BUFFER::ch_right] = 0.0;
  }
}

EFFECT_HIGHPASS::EFFECT_HIGHPASS (double cutoff) {
  add_parameter("cutoff-freq");

  set_parameter(1, cutoff);
}

void EFFECT_HIGHPASS::set_parameter(int param, double value) {
  switch (param) {
  case 1: 
    cutOffFreq = value;
    C = tan(M_PI * cutOffFreq / (double)SAMPLE_BUFFER::sample_rate);
    constant_a(0, 1.0 / (1.0 + sqrt(2.0) * C + C * C));
    constant_a(1, -2.0 * constant_a(0));
    constant_a(2, constant_a(0));
    constant_b(0, 2 * (C * C - 1.0) * constant_a(0));
    constant_b(1, (1.0 - sqrt(2.0) * C + C * C) * constant_a(0));
    break;
  }
}

double EFFECT_HIGHPASS::get_parameter(int param) const { 
  switch (param) {
  case 1: 
    return(cutOffFreq);
  }
  return(0.0);
}

EFFECT_INVERSE_COMB_FILTER::EFFECT_INVERSE_COMB_FILTER (int delay_in_samples, double radius) {
  // 
  // delay in number of samples
  // circle radius
  //
  add_parameter("delay");
  add_parameter("radius");

  set_parameter(1, (double)delay_in_samples);
  set_parameter(2, radius);
   
  MESSAGE_ITEM otemp;
  otemp.setprecision(2);
  otemp << "FX:    Inverse comb filter enabled with delay of ";
  otemp << C << " samples and circle radius of " << D << ".";
  ecadebug->msg(otemp.to_string());
}

void EFFECT_INVERSE_COMB_FILTER::set_parameter(int param, double value) {
  switch (param) {
  case 1: 
    C = value;
    laskuri = 0.0;
    break;
  case 2: 
    D = value;
    break;
  }
}

double EFFECT_INVERSE_COMB_FILTER::get_parameter(int param) const { 
  switch (param) {
  case 1: 
    return(C);
  case 2: 
    return(D);
  }
  return(0.0);
}

void EFFECT_INVERSE_COMB_FILTER::process(SAMPLE_BUFFER::sample_type *insample) {
  buffer.push_back(*insample);
    
  if (laskuri >= C) {
    insample->put_left(insample->get_left() - (pow(D, C) * (buffer.front().get_left())));
    insample->put_right(insample->get_right() - (pow(D, C) * (buffer.front().get_right())));
    buffer.pop_front();
  }
  else {
//    insample->put_left(insample->get_left() / 2.0);
//    insample->put_right(insample->get_right() / 2.0);
    laskuri++;
  }
}

EFFECT_LOWPASS::EFFECT_LOWPASS (double cutoff) {
  add_parameter("cutoff-freq");

  set_parameter(1, cutoff);
 
}

void EFFECT_LOWPASS::set_parameter(int param, double value) {
  switch (param) {
  case 1: 
    set_cutoff(value, SAMPLE_BUFFER::sample_rate);
    break;
  }
}

double EFFECT_LOWPASS::get_parameter(int param) const { 
  switch (param) {
  case 1: 
    return(cutOffFreq);
  }
  return(0.0);
}

void EFFECT_LOWPASS::set_cutoff(double value, long int srate) {
  cutOffFreq = value;
  C = 1.0 / tan(M_PI * cutOffFreq / (double)srate);
  constant_a(0, 1.0 / (1.0 + sqrt(2.0) * C + C * C));
  constant_a(1, 2.0 * constant_a(0));
  constant_a(2, constant_a(0));
  constant_b(0, 2 * (1.0 - C * C) * constant_a(0));
  constant_b(1, (1.0 - sqrt(2.0) * C + C * C) * constant_a(0));
}

EFFECT_LOWPASS_SIMPLE::EFFECT_LOWPASS_SIMPLE (double cutoff) {
  add_parameter("cutoff-freq");

  set_parameter(1, cutoff);
}

void EFFECT_LOWPASS_SIMPLE::set_parameter(int param, double value) {
  switch (param) {
  case 1: 
    cutOffFreq = value;
    A = 2.0 * M_PI * cutOffFreq / SAMPLE_BUFFER::sample_rate;
    B = exp(-A / SAMPLE_BUFFER::sample_rate);
    break;
  }
}

double EFFECT_LOWPASS_SIMPLE::get_parameter(int param) const { 
  switch (param) {
  case 1: 
    return(cutOffFreq);
  }
  return(0.0);
}

void EFFECT_LOWPASS_SIMPLE::process(SAMPLE_BUFFER::sample_type *insample) {
  tempin = *insample;
  temphist = outhist;
  outhist = tempin;

  tempin.multiply(A * 0.5);
  temphist.multiply(B * 0.5);

  insample->operator=(tempin + temphist);
}

EFFECT_RESONANT_BANDPASS::EFFECT_RESONANT_BANDPASS (double centerf,
						    double w) 
  : outhist(2)
{
  add_parameter("center-freq");
  add_parameter("width");

  set_parameter(1, centerf);
  set_parameter(2, w);

  for(int n = 0; n < 2; n++) outhist[n].make_silent();


  MESSAGE_ITEM otemp;
  otemp.setprecision(2);
  otemp << "FX:    Resonant bandpass filter enabled, center frequency is ";
  otemp << center << " and width " << width << ".";
  ecadebug->msg(otemp.to_string());
}

void EFFECT_RESONANT_BANDPASS::set_parameter(int param, double value) {
  switch (param) {
  case 1: 
    center = value;
    break;
  case 2: 
    if (value != 0) width = value;
    else width = center / 2.0;
    break;
  }
  R = 1.0 - M_PI * width / (double)SAMPLE_BUFFER::sample_rate;
  c = R * R;
  pole_angle = acos((2.0 * R) / (1.0 + c) * cos(center));
  a = (1.0 - c) * sin(pole_angle);
  b = 2.0 * R * cos(center);
}

double EFFECT_RESONANT_BANDPASS::get_parameter(int param) const { 
  switch (param) {
  case 1: 
    return(center);
  case 2: 
    return(width);
  }
  return(0.0);
}

void EFFECT_RESONANT_BANDPASS::process(SAMPLE_BUFFER::sample_type *insample) {
  insample->put_left(a * insample->get_left()  + b * outhist[0].get_left() * - c * outhist[1].get_left() - outhist[1].get_left());
  insample->put_right(a * insample->get_right() + b * outhist[0].get_right() - c * outhist[1].get_right() - outhist[1].get_right());
  
  outhist[1] = outhist[0];
  outhist[0] = *insample;
}

EFFECT_RESONANT_LOWPASS::EFFECT_RESONANT_LOWPASS (double co, double
						  res, double g) 
  : outhist(4), newhist(2), ProtoCoef(2), Coef(2)
{
  add_parameter("cutoff-freq");
  add_parameter("resonance");
  add_parameter("gain");

  cutoff = co;
  Q = res;

  gain_orig = gain = g;

  laskuri = 0.0;
    
  pi = 4.0 * atan(1.0);
    
  for(int n = 0; n < 4; n++) outhist[n].make_silent();
  for(int n = 0; n < 2; n++) newhist[n].make_silent();

//
// Setup filter s-domain coefficients
//

  ProtoCoef[0].a0 = 1.0;
  ProtoCoef[0].a1 = 0;
  ProtoCoef[0].a2 = 0;
  ProtoCoef[0].b0 = 1.0;
  ProtoCoef[0].b1 = 0.765367 / Q;      // Divide by resonance or Q
  ProtoCoef[0].b2 = 1.0;

  ProtoCoef[1].a0 = 1.0;
  ProtoCoef[1].a1 = 0;
  ProtoCoef[1].a2 = 0;
  ProtoCoef[1].b0 = 1.0;
  ProtoCoef[1].b1 = 1.847759 / Q;      // Divide by resonance or Q
  ProtoCoef[1].b2 = 1.0;

  szxform(0);
  szxform(1);
}

void EFFECT_RESONANT_LOWPASS::set_parameter(int param, double value) {
  switch (param) {
  case 1: 
    cutoff = value;
    break;
  case 2: 
    Q = value;
    break;
  case 3: 
    gain_orig = value;
    break;
  }
  refresh_values();
}

double EFFECT_RESONANT_LOWPASS::get_parameter(int param) const {
  switch (param) {
  case 1: 
    return(cutoff);
  case 2: 
    return(Q);
  case 3: 
    return(gain_orig);
  }
  return(0.0);
}

void EFFECT_RESONANT_LOWPASS::refresh_values(void) {
  if (cutoff == 0.0) cutoff = 0.1;
    
  gain = gain_orig;
    
  //    ProtoCoef[0].a0 = 1.0;
  ProtoCoef[0].a1 = 0;
  ProtoCoef[0].a2 = 0;
  //    ProtoCoef[0].b0 = 1.0;
  ProtoCoef[0].b1 = 0.765367 / Q;      // Divide by resonance or Q
  ProtoCoef[0].b2 = 1.0;

  //    ProtoCoef[1].a0 = 1.0;
  ProtoCoef[1].a1 = 0;
  ProtoCoef[1].a2 = 0;
  //    ProtoCoef[1].b0 = 1.0;
  ProtoCoef[1].b1 = 1.847759 / Q;      // Divide by resonance or Q
  ProtoCoef[1].b2 = 1.0;

  szxform(0);
  szxform(1);
}

void EFFECT_RESONANT_LOWPASS::szxform(int section) {
  wp = 2.0 * (double)SAMPLE_BUFFER::sample_rate * tan(pi * cutoff / (double)SAMPLE_BUFFER::sample_rate);

  // a0 and b0 are presumed to be 1, so...

  ProtoCoef[section].a2 = ProtoCoef[section].a2 / (wp * wp);
  ProtoCoef[section].a1 = ProtoCoef[section].a1 / wp;

  ProtoCoef[section].b2 = ProtoCoef[section].b2 / (wp * wp);
  ProtoCoef[section].b1 = ProtoCoef[section].b1 / wp;

  // alpha (Numerator in s-domain)
  ad = 4.0 * ProtoCoef[section].a2 * (double)SAMPLE_BUFFER::sample_rate * (double)SAMPLE_BUFFER::sample_rate + 2.0 * ProtoCoef[section].a1
    * (double)SAMPLE_BUFFER::sample_rate + ProtoCoef[section].a0;
  // beta (Denominator in s-domain)
  bd = 4.0 * ProtoCoef[section].b2 * (double)SAMPLE_BUFFER::sample_rate * (double)SAMPLE_BUFFER::sample_rate + 2.0 * ProtoCoef[section].b1
    * (double)SAMPLE_BUFFER::sample_rate + ProtoCoef[section].b0;

  /* update gain constant for this section */
  gain *= ad/bd;

  // Denominator
  Coef[section].A = (2.0 * ProtoCoef[section].b0 - 8.0 * ProtoCoef[section].b2
		     * (double)SAMPLE_BUFFER::sample_rate * (double)SAMPLE_BUFFER::sample_rate) / bd;
  // beta1
  Coef[section].B = (4.0 * ProtoCoef[section].b2 * (double)SAMPLE_BUFFER::sample_rate * (double)SAMPLE_BUFFER::sample_rate - 2.0 * ProtoCoef[section].b1
		     * (double)SAMPLE_BUFFER::sample_rate + ProtoCoef[section].b0) / bd;
  // beta2

  // Nominator
  Coef[section].C = (2.0 * ProtoCoef[section].a0 - 8.0 * ProtoCoef[section].a2
		     * (double)SAMPLE_BUFFER::sample_rate * (double)SAMPLE_BUFFER::sample_rate) / ad;
  // alpha1
  Coef[section].D = (4.0 * ProtoCoef[section].a2 * (double)SAMPLE_BUFFER::sample_rate * (double)SAMPLE_BUFFER::sample_rate - 2.0
		     * ProtoCoef[section].a1 * (double)SAMPLE_BUFFER::sample_rate + ProtoCoef[section].a0) / ad;
  // alpha2

}

void EFFECT_RESONANT_LOWPASS::process(SAMPLE_BUFFER::sample_type *insample) {

  insample->put_left(insample->get_left() * gain);
  insample->put_right(insample->get_right() * gain);

  // first section:
  // --------------
    
  // poles:
  insample->put_left(insample->get_left() - outhist[0].get_left() * Coef[0].A);
  insample->put_right(insample->get_right() - outhist[0].get_right() * Coef[0].A);

  newhist[0].put_left(insample->get_left() - outhist[1].get_left() * Coef[0].B);
  newhist[0].put_right(insample->get_right() - outhist[1].get_right() * Coef[0].B);
        
    // zeros:
  insample->put_left(newhist[0].get_left() + outhist[0].get_left() * Coef[0].C);
  insample->put_right(newhist[0].get_right() + outhist[0].get_right() * Coef[0].C);
    
  insample->put_left(insample->get_left() + outhist[1].get_left() * Coef[0].D);
  insample->put_right(insample->get_right() + outhist[1].get_right() * Coef[0].D);
        
  outhist[1] = outhist[0];
  outhist[0] = newhist[0];
        
    // second section:
    // --------------
    
    // poles:
  insample->put_left(insample->get_left() - outhist[2].get_left() * Coef[1].A);
  insample->put_right(insample->get_right() - outhist[2].get_right() * Coef[1].A);

  newhist[1].put_left(insample->get_left() - outhist[3].get_left() * Coef[1].B);
  newhist[1].put_right(insample->get_right() - outhist[3].get_right() * Coef[1].B);
        
    // zeros:
  insample->put_left(newhist[1].get_left() + outhist[2].get_left() * Coef[1].C);
  insample->put_right(newhist[1].get_right() + outhist[2].get_right() * Coef[1].C);
    
  insample->put_left(insample->get_left() + outhist[3].get_left() * Coef[1].D);
  insample->put_right(insample->get_right() + outhist[3].get_right() * Coef[1].D);
        
  outhist[3] = outhist[2];
  outhist[2] = newhist[1];
}

//  EFFECT_RESONANT_LOWPASS::EFFECT_RESONANT_LOWPASS (const
//  						  EFFECT_RESONANT_LOWPASS& x) 
//    : outhist(4), newhist(2), ProtoCoef(2), Coef(2)
//  {
//    outhist = x.outhist;
//    newhist = x.newhist;

//    for(vector<BIQUAD>::size_type p = 0; p != x.ProtoCoef.size(); p++) {
//      ProtoCoef[p].a0 = x.ProtoCoef[p].a0;
//      ProtoCoef[p].a1 = x.ProtoCoef[p].a1;
//      ProtoCoef[p].a2 = x.ProtoCoef[p].a2;
//      ProtoCoef[p].b0 = x.ProtoCoef[p].b0;
//      ProtoCoef[p].b1 = x.ProtoCoef[p].b1;
//      ProtoCoef[p].b2 = x.ProtoCoef[p].b2;
//      ++p;
//    }

//    for(vector<BIQUAD>::size_type p = 0; p != x.Coef.size(); p++) {
//      Coef[p].A = x.Coef[p].A;
//      Coef[p].B = x.Coef[p].B;
//      Coef[p].C = x.Coef[p].C;
//      Coef[p].D = x.Coef[p].D;
//      ++p;
//    }

//    cutoff = x.cutoff;
//    Q = x.Q;
//    gain = x.gain;
//    gain_orig = x.gain_orig;
//    pi = x.pi;
//    laskuri = x.laskuri;
  
//    ad = x.ad;
//    bd = x.bd;
//    wp = x.wp;
//  }

EFFECT_RESONATOR::EFFECT_RESONATOR (double centerf, double w) 
  : cona(1), conb(2), saout(2) 
{
  add_parameter("center-freq");
  add_parameter("width");

  set_parameter(1, centerf);
  set_parameter(2, w);
  
  MESSAGE_ITEM otemp;
  otemp.setprecision(2);
  otemp << "FX:    Resonator filter enabled, center frequency is ";
  otemp << center << " and width " << width << ".";
  ecadebug->msg(otemp.to_string());
}

void EFFECT_RESONATOR::set_parameter(int param, double value) 
{
  switch (param) {
  case 1: 
    center = value;
    break;
  case 2: 
    if (value != 0) width = value;
    else width = center / 2;
    break;
  }
  conb[1] = exp(-(2 * M_PI) * (width / (double)SAMPLE_BUFFER::sample_rate));
  conb[0] = (-4.0 * conb[1]) / (1.0 + conb[1]) * cos(2 * M_PI * (center / (double)SAMPLE_BUFFER::sample_rate));
  cona[0] = (1.0 - conb[1]) * sqrt(1.0 - (conb[0] * conb[0]) / (4.0 * conb[1]));
}

double EFFECT_RESONATOR::get_parameter(int param) const { 
  switch (param) {
  case 1: 
    return(center);
  case 2: 
    return(width);
  }
  return(0.0);
}

void EFFECT_RESONATOR::process(SAMPLE_BUFFER::sample_type *insample) {

  for(nm = 0; nm < SAMPLE_BUFFER::ch_count; nm++) {
    temp.put_channel(nm, cona[0] * insample->get_channel(nm)
		     - conb[0] * saout[0].get_channel(nm) - conb[1] * saout[1].get_channel(nm));
    
    saout[1].put_channel(nm, saout[0].get_channel(nm));
    saout[0].put_channel(nm, temp.get_channel(nm));
    
  }
  *insample = temp;
}
