// ------------------------------------------------------------------------
// eca-controller.cpp: Class for controlling ECA_SESSION, ECA_PROCESSOR
//                     and other such objects.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <iostream.h>
#include <fstream.h>
#include <string>
#include <vector>
#include <pthread.h>
#include <unistd.h>

#include <kvutils.h>

#include "eca-main.h"
#include "eca-session.h"
#include "eca-controller.h"

#include "eca-error.h"
#include "eca-debug.h"

string ecasound_lockfile;

ECA_CONTROLLER::ECA_CONTROLLER (ECA_SESSION* psession) {
  session = psession;
  show_prompt = false;
  engine_started = false;
  ecasound_lockfile = "/var/lock/ecasound.lck." + kvu_numtostr(getpid());
}

void ECA_CONTROLLER::command(const string& cmd) {
  vector<string> cmds = string_to_words(cmd);
  vector<string>::const_iterator p = cmds.begin();
  while (p != cmds.end()) {
    if (*p == "") continue;
  
    // ---
    // General
    // ---
    if (*p == "help"
	|| *p == "h"
	|| *p == "?") 
      {
	++p;
	if (p == cmds.end() || *p != "more") {
	  --p;
	  show_controller_help();
	}
	else if (p != cmds.end() || *p == "more") {
	  show_controller_help_more();
	}
	show_prompt = true;
      }
    else if (*p == "exec") {
      start_engine();
      ++p;
      if (p == cmds.end()) return;
      else --p;
    }
    else if (*p == "quit" || *p == "q") {
      quit();
    }
    else if (*p == "start" || *p == "t") start();
    else if (*p == "stop" || *p == "s") stop();
    else if (*p == "chain" || *p == "c") {
      ++p;
      if (p == cmds.end()) continue;
      ecasound_cqueue.push_back("c");
      ecasound_cqueue.push_back(*p);
      if (session->status() != ep_status_notready) {
	sleep(1);
	print_chain_status();
      }
      show_prompt = true;
    }
    else if (*p == "debug") {
      ++p;
      if (p == cmds.end()) continue;
      int level = atoi((*p).c_str());
      ecadebug->set_debug_level(level);
      ecadebug->msg("Debug level set to " + kvu_numtostr(level) + ".");
    }
    // ---
    // Chainsetups
    // ---
    else if (*p == "load") {
      ++p;
      if (p == cmds.end()) break;
      load_chainsetup(*p);
    }
    else if (*p == "save") {
      ++p;
      if (p == cmds.end()) break;
      save_active_chainsetup(*p);
    }
    // ---
    // Session status
    // ---
    else if (*p == "status" 
	     || *p == "st"
	     || *p == "u") {
      print_general_status();
    }
    else if (*p == "cstatus" 
	     || *p == "cs"
	     || *p == "a") {
      print_chain_status();
    }
    else if (*p == "estatus" 
	     || *p == "es"
	     || *p == "x") {
      print_effect_status();
    }
    else if (*p == "fstatus" 
	     || *p == "fs"
	     || *p == "l") {
      print_file_status();
    }
    // ---
    // Send to ECA_PROCESSOR
    // ---
    else {
      start_engine();
      ecasound_cqueue.push_back(*p);
    }

    ++p;
  }
  show_prompt = true;
}

bool ECA_CONTROLLER::prompt(void) {
  if (show_prompt == true) {
    show_prompt = false;
    return(true);
  }
  return(false);
}

void ECA_CONTROLLER::activate_chainsetup(const string& name, bool handle_errors) {
  stop();
  try {
    close_engine();
    session->set_active_setup(name);
  }
  catch (ECA_ERROR* e) {
    if (!handle_errors) throw;
    cerr << "---\n(eca-controller) FAILED: [" << e->error_section() << "] : \"" << e->error_msg() << "\"\n\n";
  }
}

string ECA_CONTROLLER::active_chainsetup(void) {
 if (session->active_chainsetup != 0)
   return(session->active_chainsetup->name());

 return("");
}

void ECA_CONTROLLER::deactivate_chainsetup(const string& name, bool handle_errors) {
  if (name != active_chainsetup()) return;
  if (name == connected_chainsetup()) {
    stop();
    close_engine();
  }    

  try {
    session->deactivate_setup();
  }
  catch (ECA_ERROR* e) {
    if (!handle_errors) throw;
    cerr << "---\n(eca-controller) FAILED: [" << e->error_section() << "] : \"" << e->error_msg() << "\"\n\n";
  }
}


void ECA_CONTROLLER::connect_active_chainsetup(void) {
  session->connect_active_setup();
}

string ECA_CONTROLLER::connected_chainsetup(void) {
  if (session->active_chainsetup != 0) {
    if (session->is_active_chainsetup_connected()) {
      return(session->active_chainsetup->name());
    }
  }

  return("");
}

void ECA_CONTROLLER::disconnect_active_chainsetup(void) {
  session->disconnect_active_setup();
}

void ECA_CONTROLLER::remove_chainsetup(const string& name, bool handle_errors) {
  if (session->active_chainsetup != 0 && name ==
      session->active_chainsetup->name()) {
    
    throw(new ECA_ERROR("ECA-CONTROLLER","Can't delete active chainsetup."));
  }

  vector<ECA_CHAINSETUP*>::iterator p = session->chainsetups.begin();
  while(p != session->chainsetups.end()) {
    if ((*p)->name() == name) {
      (*p)->disable();
      session->chainsetups.erase(p);
      break;
    }
    ++p;
  }
}

void ECA_CONTROLLER::new_chainsetup(const string& name) {
  
  ecadebug->msg("(eca-controller) Adding a new chainsetup with name \"" + name + "\".");
  
  session->add_new_setup(name);
}

void ECA_CONTROLLER::load_chainsetup(const string& name, bool handle_errors) {
  ecadebug->msg("(eca-controller) Loading a new chainsetup from file \""
		+ name + "\".");
      
  bool was_running = false;
  if (session->status() == ep_status_running) was_running = true;
  stop();
  try {
    close_engine();
    session->load_active_setup(name);
  }
  catch (ECA_ERROR* e) {
    if (!handle_errors) throw;
    cerr << "---\n(eca-controller) FAILED: [" << e->error_section() << "] : \"" << e->error_msg() << "\"\n\n";
  }
  if (was_running) ecasound_cqueue.push_back("start");
}

void ECA_CONTROLLER::save_chainsetup(const string& csetup, const string
				     filename, bool handle_errors) {
  try {
    vector<ECA_CHAINSETUP*>::iterator p = session->chainsetups.begin();
    while(p != session->chainsetups.end()) {
      if ((*p)->name() == csetup) {

	ecadebug->msg("(eca-controller) Saving active chainsetup to file \""
		      + filename + "\".");
	(*p)->save_to_file(filename);
      }
      ++p;
    }
  }
  catch (ECA_ERROR* e) {
    if (!handle_errors) throw;
    cerr << "---\n(eca-controller) FAILED: [" << e->error_section() << "] : \"" << e->error_msg() << "\"\n\n";
  }
}

const ECA_CHAINSETUP* ECA_CONTROLLER::get_chainsetup(const string& name) {
  vector<ECA_CHAINSETUP*>::const_iterator p = session->chainsetups.begin();
  while(p != session->chainsetups.end()) {
    if ((*p)->name() == name) {
      return((*p));
    }
    ++p;
    }
  return(0);
}

void ECA_CONTROLLER::save_active_chainsetup(const string& filename, bool handle_errors) {
  if (!session->is_active_chainsetup_connected()) {
    throw(new ECA_ERROR("ECA_CONTROLLER", "Can't save, no active chainsetup available."));
  }
  save_chainsetup(session->active_chainsetup->name(), 
		  filename,
		  handle_errors);
}

void ECA_CONTROLLER::start(void) {
  if (session->status() == ep_status_running) return;
  ecadebug->control_flow("Controller/Processing started");

  if (session->status() == ep_status_notready) {
    start_engine();
  }

  if (is_engine_ready() == false) {
    ecadebug->msg("(eca-controller) Can't execute engine; active chainsetup is not valid.");
    return;
  }  

  ecasound_cqueue.push_back("t");
  long int usleep_count = 5000;
  while(session->status() != ep_status_running) {
    usleep(++usleep_count);
    if (session->status() == ep_status_finished) break;
    if (usleep_count > 5000000) {
      throw(new ECA_ERROR("ECA_CONTROLLER", "start() failed, engine has not responded during the last 5 seconds...", ECA_ERROR::stop));
    }
  }
}

void ECA_CONTROLLER::stop(void) {
  if (session->status() != ep_status_running) return;
  ecadebug->control_flow("Controller/Processing stopped");
  ecasound_cqueue.push_back("s");
  int usleep_count = 1000;
  while(session->status() == ep_status_running) {
    usleep(++usleep_count); 
    if (usleep_count > 5000000) {
      throw(new ECA_ERROR("ECA_CONTROLLER", "start() failed, engine has not responded during the last 5 seconds...", ECA_ERROR::stop));
    }
  }
}

void ECA_CONTROLLER::quit(void) {
  close_engine();
  int n = ECA_QUIT;
  throw(n);
}

void ECA_CONTROLLER::start_engine(bool ignore_lock) {
  if (engine_started == true) return;

  if (session->active_chainsetup == 0) {
    ecadebug->msg("(eca-controller) Can't execute engine; no active chainsetup.");
    return;
  }

  session->connect_active_setup();

  if (is_engine_ready() == false) {
    ecadebug->msg("(eca-controller) Can't execute engine; active chainsetup is not valid.");
    return;
  }

  engine_started = true;

  ifstream fin(ecasound_lockfile.c_str());
  if (!fin || ignore_lock)
    start_normal_thread(session, retcode, &th_cqueue);
  else {
    MESSAGE_ITEM mitem;
    mitem << "(eca-controller) Can't execute; processing module already running!" << 'c' << "\n";
    //    cerr << "\"" << mitem.to_string() << "\"";
    ecadebug->msg(1,mitem.to_string());
  }
}

void ECA_CONTROLLER::close_engine(void) {
  if (!engine_started) return;
  ecasound_cqueue.push_back("end");
  ifstream fin(ecasound_lockfile.c_str());
  while(fin) {
    fin.close();
    ecadebug->msg(1, "(eca-controller) Waiting for the processing thread...");
    usleep(1000);
    fin.open(ecasound_lockfile.c_str());
  }
  ecasound_cqueue.flush();
  engine_started = false;
  //  fin.close();
}

void ECA_CONTROLLER::print_general_status(void) {
  ecadebug->control_flow("Controller/General Status");
  MESSAGE_ITEM st_info_string;

  if (session->active_chainsetup != 0) {
    st_info_string << "Buffersize is " << session->active_chainsetup->buffersize << " samples.\n";
    if (session->active_chainsetup->raisepriority) st_info_string << "Priority risen.\n";
  }
  else 
    st_info_string << "No chainsetups are connected to the engine.\n";

  st_info_string << "Internal sampling rate is " << SAMPLE_BUFFER::sample_rate << " samples per second.\n";

  if (session->multitrack_mode) st_info_string << "Multitrack-mode enabled.\n";
  else st_info_string << "Multitrack-mode disabled.\n";
  if (session->loop_active)
    st_info_string << "Looping enabled. Starts at "
		   << session->loop_start_pos
		   << " and ends at "
		   << session->loop_end_pos
		   << ".\n";
  else
    st_info_string << "Looping disabled.\n";
  if (session->sfx) st_info_string << "Effect section enabled.";
  else st_info_string << "Effect section disabled.";

  ecadebug->msg(st_info_string.to_string());
}

void ECA_CONTROLLER::print_chain_status(void) {
  ecadebug->control_flow("Controller/Chain Status");

  if (!is_engine_ready()) return;

  MESSAGE_ITEM mitem;
  vector<CHAIN*>::const_iterator chain_citer;
  vector<CHAIN_OPERATOR*>::const_iterator chainop_citer;

  for(chain_citer = session->chains->begin(); chain_citer != session->chains->end();) {
    mitem << "Chain \"" << (*chain_citer)->name() << "\" [";
    if ((*chain_citer)->is_enabled()) mitem << "on] ";
    else mitem << "off] ";
    if (session->sfx) {
      for(chainop_citer = (*chain_citer)->chainops.begin(); chainop_citer != (*chain_citer)->chainops.end();) {
	mitem << "\"" << (*chainop_citer)->name() << "\"";
	++chainop_citer;
	if (chainop_citer != (*chain_citer)->chainops.end()) mitem << " -> ";   
      }
    }
    ++chain_citer;
    if (chain_citer != session->chains->end()) mitem << "\n";
  }
  ecadebug->msg(mitem.to_string());
}

void ECA_CONTROLLER::print_effect_status(void) {
  ecadebug->control_flow("Controller/Effect Status");
  MESSAGE_ITEM mitem;
  string st_info_string;
  vector<CHAIN*>::const_iterator chain_citer;
  vector<CHAIN_OPERATOR*>::const_iterator chainop_citer;

  if (!is_engine_ready()) return;

  for(chain_citer = session->chains->begin(); chain_citer != session->chains->end();) {
    chainop_citer = (*chain_citer)->chainops.begin();
    while(chainop_citer != (*chain_citer)->chainops.end()) {
      mitem << "FX " << (*chainop_citer)->name();
      //      mitem << " with params \"" << (*chainop_citer)->params() << "\"";
      mitem << ", chain " << (*chain_citer)->name() << ": ";
      for(int n = 0; n < (*chainop_citer)->number_of_params(); n++) {
	mitem << (*chainop_citer)->get_parameter_name(n + 1);
	mitem << " ";
	mitem << kvu_numtostr((*chainop_citer)->get_parameter(n + 1));
	if (n + 1 < (*chainop_citer)->number_of_params()) mitem <<  ", ";
      }
      st_info_string = (*chainop_citer)->status();
      if (st_info_string != "") {
	mitem << "FX-status:\n" << st_info_string;
      }
      ++chainop_citer;
      if (chainop_citer != (*chain_citer)->chainops.end()) mitem << "\n";
      //      if (chainop_citer != (*chain_citer)->chainops.end()) mitem << " ";   
    }
    ++chain_citer;
    //    if (chain_citer == session->chains->end()) mitem << "\n";
  }
  if (mitem.to_string() != "") ecadebug->msg(mitem.to_string());
  st_info_string = "";
}

void ECA_CONTROLLER::print_file_status(void) {
  ecadebug->control_flow("Controller/File status");

  if (!is_engine_ready()) return;

  string st_info_string;
  vector<AUDIO_IO*>::const_iterator adev_citer;
  vector<AUDIO_IO*>::size_type adev_sizet = 0;

  adev_citer = session->inputs->begin();

  while(adev_citer != session->inputs->end()) {
    st_info_string += "Infile \"";
    st_info_string += (*adev_citer)->label();
    st_info_string += "\", connected to chains \"";
    vector<string> temp = session->get_connected_chains_to_input(adev_sizet);
    vector<string>::const_iterator p = temp.begin();
    while (p != temp.end()) {
      st_info_string += *p; 
      ++p;
      if (p != temp.end())  st_info_string += ",";
    }
    st_info_string += "\": ";
    st_info_string += (*adev_citer)->status();
    st_info_string += "\n";
    ++adev_sizet;
    ++adev_citer;
  }

  adev_sizet = 0;
  adev_citer = session->outputs->begin();

  while(adev_citer != session->outputs->end()) {
    st_info_string += "Outfile \"";
    st_info_string += (*adev_citer)->label();
    st_info_string += "\", connected to chains \"";
    vector<string> temp = session->get_connected_chains_to_output(adev_sizet);
    vector<string>::const_iterator p = temp.begin();
    while (p != temp.end()) {
      st_info_string += *p; 
      ++p;
      if (p != temp.end())  st_info_string += ",";
    }
    st_info_string += "\": ";
    st_info_string += (*adev_citer)->status();
    ++adev_sizet;
    ++adev_citer;
    if (adev_sizet < session->outputs->size()) st_info_string += "\n";
  }
  ecadebug->msg(st_info_string);
}

bool ECA_CONTROLLER::is_engine_ready(void) const {
  if (session->active_chainsetup == 0) return(false);
  if (session->is_active_chainsetup_connected() == false) return(false);
  return(session->active_chainsetup->is_valid());
}

bool ECA_CONTROLLER::is_running(void) {
  if (session->status() == ep_status_running) return(true);
  else return(false);
}

string ECA_CONTROLLER::engine_status(void) {
  switch(session->status()) {
  case ep_status_running: 
    {
    return("running"); 
    }
  case ep_status_stopped: 
    {
    return("stopped"); 
    }
  case ep_status_finished:
    {
    return("finished"); 
    }
  case ep_status_notready: 
    {
    return("not ready"); 
    }
  default: 
    {
    return("unknown status"); 
    }
  }
}

string ECA_CONTROLLER::connected_chains_input(CHAIN::aio_id_type aiod)
{
  if (session->active_chainsetup == 0) return("");

  vector<string> t = session->get_connected_chains_to_input(aiod);
  string out = "";
  vector<string>::const_iterator p = t.begin();
  while(p != t.end()) {
    out += *p;
    ++p;
    if (p != t.end()) out += ",";
  }
  return(out);
}

string ECA_CONTROLLER::connected_chains_output(CHAIN::aio_id_type
					       aiod) {
  if (session->active_chainsetup == 0) return("");

  vector<string> t = session->get_connected_chains_to_output(aiod);
  string out = "";
  vector<string>::const_iterator p = t.begin();
  while(p != t.end()) {
    out += *p;
    ++p;
    if (p != t.end()) out += ",";
  }
  return(out);
}

vector<string> ECA_CONTROLLER::connected_chains(const string&
						filename) {
  if (session->active_chainsetup == 0) return(*(new vector<string> (0)));
  return(session->active_chainsetup->get_connected_chains_to_iodev(filename));
}

void ECA_CONTROLLER::add_chain(const string& name) { 
  if (session->active_chainsetup == 0) return;

  vector<string> t;
  t.push_back(name);

  stop();
  close_engine();
  session->active_chainsetup->add_new_chains(t);
}

void ECA_CONTROLLER::remove_chain(const string& name) { 
  if (session->active_chainsetup == 0) return;
  stop();
  close_engine();
  //  cerr << "A";
  session->active_chainsetup->remove_chain(name);
  //  cerr << "B";
  //  cerr << "C";
  //  cerr << "D";
}

void ECA_CONTROLLER::set_active_chains(const vector<string>& chains) {
  if (session->active_chainsetup == 0) return;
  session->active_chainsetup->set_active_chains(chains);
}

void ECA_CONTROLLER::set_audio_format(int bits,
				      int channels, 
				      long int srate) { 
  if (session->active_chainsetup == 0) return;
  string format;
  format = "-f:";
  format += kvu_numtostr(bits);
  format += ",";
  format += kvu_numtostr(channels);
  format += ",";
  format += kvu_numtostr(srate);

  session->active_chainsetup->interpret_setup_state(format);
}

void ECA_CONTROLLER::set_audio_format(const AIO_PARAMS* format) {
  set_audio_format(static_cast<int>(format->bits), 
		   static_cast<int>(format->channels), 
		   static_cast<long int>(format->srate));
}

void ECA_CONTROLLER::get_audio_format(const string& name, AIO_PARAMS* format) {
  vector<AUDIO_IO*>::size_type p = 0;
  ECA_CHAINSETUP* q = session->active_chainsetup;
  if (q == 0) return;
  
  for(p = 0; p != q->inputs.size(); p++) {
    if (q->inputs[p]->label() == name) {
      *format = q->inputs[p]->format();
    }
  }

  for(p = 0; p != q->outputs.size(); p++) {
    if (q->outputs[p]->label() == name) {
      *format = q->outputs[p]->format();
    }
  }
}

void ECA_CONTROLLER::set_explicit_format(bool enabled) {
  if (session->active_chainsetup == 0) return;
  
  session->active_chainsetup->set_explicit_format(enabled);
}

void ECA_CONTROLLER::add_audio_device(const string& filename,
				      DIRECTION dir) { 
  if (session->active_chainsetup == 0) return;

  stop();
  close_engine();

  if (dir == input)
    session->active_chainsetup->interpret_audioio_device("-i",
							 filename);
  else
    session->active_chainsetup->interpret_audioio_device("-o",
							 filename);
}

AUDIO_IO* ECA_CONTROLLER::get_audio_device(const string& csname, 
						  const string& name) {

  const ECA_CHAINSETUP* q = get_chainsetup(csname);
  if (q == 0) return(0);
  
  vector<AUDIO_IO*>::size_type p = 0;
  for(p = 0; p != q->inputs.size(); p++) {
    if (q->inputs[p]->label() == name) {
      return(q->inputs[p]);
    }
  }

  for(p = 0; p != q->outputs.size(); p++) {
    if (q->outputs[p]->label() == name) {
      return(q->outputs[p]);
    }
  }

  return(0);
}

void ECA_CONTROLLER::add_default_output(void) {
  if (session->active_chainsetup == 0) return;
  stop();
  close_engine();

  session->active_chainsetup->interpret_audioio_device("-o", session->ecaresources.resource("default-output"));
}

void ECA_CONTROLLER::remove_audio_device(const string& filename) { 
  if (session->active_chainsetup == 0) return;
  stop();
  close_engine();
  session->disconnect_active_setup();

  session->active_chainsetup->remove_audio_device(filename);
}

void ECA_CONTROLLER::attach_iodev_to_active_chains(const string& filename) {
  if (session->active_chainsetup == 0) return;

  stop();
  close_engine();
  session->disconnect_active_setup();

  session->active_chainsetup->attach_iodev_to_active_chains(filename);
}

void ECA_CONTROLLER::add_chain_operator(const string& chainop_params) { 
  if (session->active_chainsetup == 0) return;

  stop();
  close_engine();

  session->active_chainsetup->interpret_chain_operator(chainop_params);
}

void ECA_CONTROLLER::add_chain_operator(CHAIN_OPERATOR* cotmp) { 
  if (session->active_chainsetup == 0) return;
  stop();
  close_engine();

  session->active_chainsetup->add_chainop(cotmp);
}

const CHAIN_OPERATOR* ECA_CONTROLLER::get_chain_operator(const string& chain, int
				   chainop_id) {
  vector<CHAIN*>::size_type p = 0;
  ECA_CHAINSETUP* q = session->active_chainsetup;
  if (q == 0) return(0);

  for(p = 0; p != q->chains.size(); p++) {
    if (q->chains[p]->name() == chain) {
      if (chainop_id < (int)q->chains[p]->chainops.size())
	return(q->chains[p]->chainops[chainop_id]);
      else
	return(0);
    }
  }
  return(0);
}

void ECA_CONTROLLER::remove_chain_operator(const string& chain,
					     int chainop_id) { }
void ECA_CONTROLLER::add_general_controller(const string& gcontrol_params) { }

void ECA_CONTROLLER::set_buffersize(int bsize) { 
  if (session->active_chainsetup == 0) return;
  session->active_chainsetup->buffersize = bsize; 
}

void ECA_CONTROLLER::toggle_raise_priority(bool v) { 
  if (session->active_chainsetup == 0) return;
  session->active_chainsetup->raisepriority = v; 
}


void show_controller_help(void) {
  MESSAGE_ITEM mitem; 

  mitem << "\n-------------------------------------------------------------------";
  mitem << "\n(qt)ecasound - command reference\n";

  mitem << "\n'q' - Quits ecasound";
 
  mitem << "\n'start', 't' - Processing is started (play)";
 
  mitem << "\n'stop', 's' - Stops processing"; 
 
  mitem << "\n'rewind time-in-seconds', 'rw time-in-seconds' - Rewind";
 
  mitem << "\n'forward time-in-seconds', 'fw time-in-seconds' - Forward";
 
  mitem << "\n'setpos time-in-seconds' - Sets the current position to 'time-in-seconds' seconds from the beginning.";
 
  mitem << "\n'status','st','u' - General status info";
 
  mitem << "\n'cstatus','cs','a' - Status info about the effect chains";
 
  mitem << "\n'estatus', 'es','x' - Status info about effects and controller envelopes";
 
  mitem << "\n'fstatus', 'fs','l' - Status info about open files and devices";
 
  mitem << "\n'sfx' - Enable/disable the sound effects section";
 
  mitem << "\n'chain chainname', 'c chainname' - Enable/disable the the chain 'chainname'";
 
  mitem << "\n'help more' - More commands";

  ecadebug->msg(mitem.to_string());
}

void show_controller_help_more(void) {
  MESSAGE_ITEM mitem; 

  mitem << "\n-------------------------------------------------------------------";
  mitem << "\n(qt)ecasound - command reference (more)\n";

  mitem << "\n'debug debug_level' - Set debug level.";
  mitem << "\n'exec' - Starts the processing engine. Normally you don't need to use this.";
  mitem << "\n'end' - Exits the processing engine, but doesn't exit the interactive mode. Normally you don't need to use this.";

  mitem << "\n'loop' - Start/stop looping";
 
  mitem << "\n'loop_start seconds_from_start' - Set the start point for looping";
  mitem << "\n'loop_end seconds_from_start' - Set the end point for looping";

  mitem << "\n'load file_name' - Load a chainsetup from file 'file_name' and set it active.";

  mitem << "\n'save file_name' - Save the active chainsetup to file 'file_name'.";

  ecadebug->msg(mitem.to_string());
}

void start_normal_thread(ECA_SESSION* param, int retcode, pthread_t* th_ecasound_cqueue) {
  retcode = pthread_create(th_ecasound_cqueue, NULL, start_normal, (void*)param);
  if (retcode != 0)
    throw(new ECA_ERROR("ECA-CONTROLLER", "Unable to create a new thread (start_normal)."));
}

void* start_normal(void* param) {
  ofstream fout(ecasound_lockfile.c_str());
  fout.close();
  ecadebug->msg(1,"(eca-controller) Engine-thread pid: " + kvu_numtostr(getpid()));
  start_normal((ECA_SESSION*)param);
  remove(ecasound_lockfile.c_str());
  return(0);
}

void start_normal(ECA_SESSION* param) {
  try {
    ECA_PROCESSOR epros (param);
    epros.exec();
  }
  catch(ECA_ERROR* e) {
    cerr << "---\n(eca-controller) ERROR: [" << e->error_section() << "] : \"" << e->error_msg() << "\"\n\n";
  }
  catch(...) {
    cerr << "---\n(eca-controller) Caught an unknown exception!\n";
  }
}
