#ifndef _DYNAMIC_PARAMETERS_H
#define _DYNAMIC_PARAMETERS_H

#include <map>
#include <string>

/**
 * Virtual class that provides a system for dynamically controlling 
 * a set of parameters. Supports naming of parameters, minimum and 
 * maximum values and validity checks.
 * @author Kai Vehmanen
 */
class DYNAMIC_PARAMETERS {
 
  bool param_lowhigh_enabled;

  map<string,int> param_map;
  map<int,string> param_revmap;
  map<int,double> param_revmap_low;
  map<int,double> param_revmap_high;

 protected:

  /**
   * Add a new parameter. Derived classes should use this routine 
   * to specify their parameters.
   */
  void add_parameter(const string& name, double low_limit = 0.0, double high_limit = 0.0) { 
    int id = (int)(param_map.size() + 1);
    param_map[name] = id;
    param_revmap[id] = name;
    param_revmap_low[id] = low_limit;
    param_revmap_high[id] = high_limit;
    if (low_limit == 0.0 &&
	high_limit == 0.0)
      param_lowhigh_enabled = false;
    else
      param_lowhigh_enabled = true;
  }

 public:

  /**
   * Get parameter id number.
   * @param name parameter name
   */
  int get_parameter_id(const string& name) { return(param_map[name]); }

  /**
   * Get number of of parameters.
   */
  int number_of_params(void) const { return(param_map.size()); }

  /**
   * Get parameter name. 
   * @param id parameter id number
   * 
   */
  const string& get_parameter_name(int id) { return(param_revmap[id]); }

  /**
   * Get the lowest valid value is for this parameter.
   * @param param parameter id
   */
  double get_parameter_low_limit(int param) { return(param_revmap_low[param]); }

  /**
   * Get the highest valid value is for this parameter.
   * @param param parameter id
   */
  double get_parameter_high_limit(int param) { return(param_revmap_high[param]); }

  /**
   * Whether low and high limits are set for all parameters.
   */
  bool parameter_limits_enabled(void) { return(param_lowhigh_enabled); }

  /**
   * Set parameter value. 
   * @param param parameter id
   * @param value new value
   */
  virtual void set_parameter(int param, double value) { }

  /**
   * Get parameter value. 
   * @param param parameter id
   */
  virtual double get_parameter(int param) { return(0.0); }

  /**
   * If implemented in subclasses, this can be used for doing 
   * sanity checks on parameter values before using them.
   * @param param parameter id
   * @param value value to check
   */
  virtual bool is_valid_parameter_value(int param, double value) { return(true); }

  DYNAMIC_PARAMETERS (void) : param_lowhigh_enabled(false) { }
  virtual ~DYNAMIC_PARAMETERS (void) { }
};

#endif
