#ifndef _AUDIOIO_H
#define _AUDIOIO_H

#include <vector>
#include <stdio.h>
#include <math.h>
#include <string>

#include <kvutils.h>

#include "eca-audioposition.h"
#include "eca-debug.h"

class SAMPLE_BUFFER;

/**
 * Input/Output mode
 *
 * <ul>
 * <li> <b>si_read</b> 
 * Device is opened for input. If opening a file, 
 * it must exist.
 * <li> <b>si_write</b> 
 * Device is opened for output. If opening a file and
 * and output exists, it is first truncated.
 * <li> <b>si_readwrite</b> 
 * Device is opened for both reading and writing. If
 * opening a file, a new file is created if needed. 
 * When switching from read to write or vica versa,
 * position should be reset before using the device.
 * </ul>
 */
enum SIMODE { si_read, si_write, si_readwrite };

/**
 * Sample format parameters
 */
class AIO_PARAMS {
 public:

  char channels;
  long int srate;
  long int byte_second;         // bytes per seconds
  short int align;              // the size of one sample in bytes
  char bits;

  AIO_PARAMS (void) { 
    channels = 2;
    srate = 44100;
    bits = 16;
    byte_second = srate * channels * bits / 8;
    align = channels * bits / 8;
  }
};

/**
 * Virtual base for all audio I/O classes (files, audio devices,
 * sound producing program modules, etc.)
 * @author Kai Vehmanen
 */
class AUDIO_IO : public ECA_AUDIOPOSITION {

 private:

  AIO_PARAMS sf_fmt;
  SIMODE si_mode;
  bool fin;
  string id_label;
  int bsize;
  bool open_rep;
    
 public:

  /**
   * Write a (@ref SAMPLE_BUFFER) object to the device.
   */
  virtual void write_buffer(SAMPLE_BUFFER* t) = 0;

  /**
   * Read a (@ref SAMPLE_BUFFER) object from the device. 
   */
  virtual void read_buffer(SAMPLE_BUFFER* t) = 0;
    
  /**
   * Open device (possibly in exlusive mode).
   */
  virtual void open_device(void) { }

  /**
   * Close device. After a call to this method, device must be
   * free so that other processes can use it.
   */
  virtual void close_device(void) { }

  /**
   * Optional status string. 
   */
  virtual string status(void) const;


  /**
   * With this method, derived class can specify whether it
   * represents a realtime device? Here, a realtime device...
   * <ul>
   * <li> is disabled after device is opened
   * <li> is enabled with the rt_activate() method
   * <li> once enabled, will handle I/O at a constant speed
   *      based on the sample format paremeters
   * <li> must be stopped with rt_stop() method
   * </ul>
   */
  virtual bool is_realtime(void) const = 0;

  /**
   * Stop realtime device. I/O is not allowed after this call. This
   * should be used when device will not be used in a while.
   */
  virtual void rt_stop(void) { } 

  /** 
   * Prepare realtime device for processing. I/O is allowed. This is 
   * useful for preloading realtime output devices.
   */
  virtual void rt_ready(void) { }

  /**
   * Activates the devices. Underruns will occur, if the calling
   * program can't handle the realtime processing speed of this 
   * device. Overruns are handled by blocking write_buffer() calls
   * if necessary.
   */
  virtual void rt_activate(void) { }

  /**
   * Get a string containing info about sample format parameters.
   */
  string format_info(void) const;

  /**
   * Returns info about the I/O mode of this device. 
   * @see SIMODE
   */
  inline const SIMODE io_mode(void) const { return(si_mode); }

  /**
   * Returns info about the sample format parameters.
   * @see AIO_PARAMS
   */
  inline const AIO_PARAMS& format(void) const { return(sf_fmt); }

  /**
   * The device name (usually set to device/file name).
   */
  const string& label(void) const { return(id_label); }

  /**
   * Internal buffersize in samples.
   */
  inline int buffersize(void) const { return(bsize); }

  /**
   * Set internal buffersize. 
   * @param value new buffer size in samples
   */
  void buffersize(int value);

  /**
   * Is device's state finished? If opened in 'si_read' mode, 
   * finished means that end of streams has been reaches. 
   * If opened in 'si_write' or 'si_readwrite' modes, 
   * finished means that an error has occures (no space left, etc).
   * @see SIMODE
   */
  inline bool finished(void) const { return(fin); }

  /**
   * Has device been opened (with open_device())?
   */
  inline bool is_open(void) const { return(open_rep); }

  // --
  // Constructors and destructors
  // --
  virtual ~AUDIO_IO(void) { }
  AUDIO_IO(const string& name, const SIMODE mode, const
		  AIO_PARAMS& fmt, int b = 0)
    { 
      label(name);
      io_mode(mode);
      format(fmt);
      buffersize(b);
    }
 
  AUDIO_IO(void) {
    label("unknown");
    io_mode(si_read);
    format(AIO_PARAMS()); // use default params
    buffersize(0);
  }
   
 protected:

  virtual AUDIO_IO* clone(void) = 0;

  // --
  // Format setup
  // --
  /**
   * Set IO-mode.
   */
  void io_mode(SIMODE newmode) { si_mode = newmode; }

  /**
   * Set sample format parameters.
   */
  void format(const AIO_PARAMS& sffmt);

  /**
   * Sets device name.
   */
  void label(const string& newlabel) { id_label = newlabel; }

  /**
   * Set device state to enabled or disabled. 
   */
  void set_open_state(bool value) { open_rep = value; }

  /** 
   * Set device state to (un)finished.
   */
  void finished(bool value) { fin = value; }

 private:

  void fix_format(AIO_PARAMS& format) {
    format.byte_second = format.channels * format.bits * format.srate / 8;
    format.align = format.channels * format.bits / 8;
  }
};


/**
 * Virtual base for all file I/O classes.
 * @author Kai Vehmanen
 */
class AUDIO_IO_FILE : public AUDIO_IO {

 public:

  virtual bool is_realtime(void) const { return(false); }

  AUDIO_IO_FILE(const string& name, const SIMODE mode, const
		AIO_PARAMS& fmt, int b = 0) :
    AUDIO_IO(name, mode, fmt, b) { }
  AUDIO_IO_FILE(void) { }

  virtual ~AUDIO_IO_FILE(void) { }
};


/**
 * Virtual base for all audio I/O devices.
 * @author Kai Vehmanen
 */
class AUDIO_IO_DEVICE : public AUDIO_IO {
 
 public:

  virtual bool is_realtime(void) const { return(true); }

  AUDIO_IO_DEVICE(const string& name, const SIMODE mode, const
		  AIO_PARAMS& fmt, int b = 0) :
    AUDIO_IO(name, mode, fmt, b) { }
  AUDIO_IO_DEVICE(void) { }

  virtual ~AUDIO_IO_DEVICE(void) { }
};

class NULLFILE : public AUDIO_IO_FILE {
  //
  // Null AUDIO_IO_DEVICE
  //
 public:

  void read_buffer(SAMPLE_BUFFER* t) { }
  void write_buffer(SAMPLE_BUFFER* t) { }
 

  NULLFILE(void) {
    label("null");
    finished(false);
  }
  NULLFILE(const SIMODE mode) {
    io_mode(mode);
    label("null");
    finished(false);
  }
  ~NULLFILE(void) { }
  NULLFILE* clone(void) { return new NULLFILE(*this); }
};

#endif



