#ifndef _AUDIOFX_TIMEBASED_H
#define _AUDIOFX_TIMEBASED_H

#include <vector>
#include <string>

#include "audiofx.h"

typedef deque<SAMPLE_BUFFER::sample_type> SINGLE_BUFFER;

/**
 * Base class for time-based effects (delays, reverbs, etc).
 */
class EFFECT_TIME_BASED : public EFFECT_BASE {

 protected:
    
  double dtime;        // delay time in samples
  double laskuri;
  
 public:

  EFFECT_TIME_BASED (void) : dtime(0.0), laskuri(0.0) { }
  virtual EFFECT_TIME_BASED* clone(void) = 0;
};

/** 
 * Delay effect.
 */
class EFFECT_DELAY : public EFFECT_TIME_BASED {

private:

    SAMPLE_BUFFER::sample_type temp, temp2;     // temporary values
    int nm2; // temp-value

    double surround;
    double dnum;
    double mix;

    vector<SINGLE_BUFFER> buffer;

public:

    string name(void) { return("Delay"); }

    double get_parameter(int param);
    void set_parameter(int param, double value);

    void process(SAMPLE_BUFFER::sample_type *insample);
    double get_delta_in_samples(void) { return(dnum * dtime); }

    EFFECT_DELAY* clone(void)  { return new EFFECT_DELAY(*this); }
    EFFECT_DELAY (double delay_time, int surround_mode, int num_of_delays, double mix_percent);
    EFFECT_DELAY (void) {
      add_parameter("delay_time");
      add_parameter("surround_mode");
      add_parameter("number_of_delays");
      add_parameter("mix_%");
    }
};

/*
 * Transforms a mono signal to stereo using a panned delay signal.
 * Suitable delays values range from 1 to 40 milliseconds. 
 */
class EFFECT_FAKE_STEREO : public EFFECT_TIME_BASED {

  SAMPLE_BUFFER::sample_type temp;   // temp-values

  deque<SAMPLE_BUFFER::sample_type> buffer;

public:

    string name(void) { return("Fake stereo"); }

    double get_parameter(int param);
    void set_parameter(int param, double value);

    void process(SAMPLE_BUFFER::sample_type *insample);

    EFFECT_FAKE_STEREO* clone(void)  { return new EFFECT_FAKE_STEREO(*this); }
    EFFECT_FAKE_STEREO::EFFECT_FAKE_STEREO (double delay_time);
    EFFECT_FAKE_STEREO (void) {
      add_parameter("delay_time");
    }
};

/*
 * Simple reverb effect.
 */
class EFFECT_REVERB : public EFFECT_TIME_BASED {

private:
    
  SAMPLE_BUFFER::sample_type temp;  // temp-values

  deque<SAMPLE_BUFFER::sample_type> buffer;

  double surround;
  double feedback;

 public:

  string name(void) { return("Reverb"); }

  double get_parameter(int param);
  void set_parameter(int param, double value);

  void process(SAMPLE_BUFFER::sample_type *insample);
  double get_delta_in_samples(void) { return(dtime); }

  EFFECT_REVERB* clone(void)  { return new EFFECT_REVERB(*this); }
  EFFECT_REVERB (double delay_time, int surround_mode, double feedback_percent);
  EFFECT_REVERB (void) {
    add_parameter("delay_time");
    add_parameter("surround_mode");
    add_parameter("feedback_%");
  }
};

#endif


