#ifndef _AUDIOFX_H
#define _AUDIOFX_H

#include <vector>

#include <kvutils.h>

#include "eca-debug.h"
#include "eca-chainop.h"
#include "samplebuffer.h"

#include "eca-chainop.h"

/**
 * Virtual base for all audio effect classes.
 * @author Kai Vehmanen
 */
class EFFECT_BASE : public CHAIN_OPERATOR {

 public:
  virtual ~EFFECT_BASE(void) { }
};

/**
 * Virtual base for signal analyzers.
 * @author Kai Vehmanen
 */
class EFFECT_ANALYSIS : public EFFECT_BASE {

 public:
  virtual ~EFFECT_ANALYSIS(void) { }
};

/**
 * Virtual base for amplitude effects and dynamic processors.
 * @author Kai Vehmanen
 */
class EFFECT_DYNAMIC : public EFFECT_BASE {

 public:
  virtual ~EFFECT_DYNAMIC(void) { }
};

/**
 * Amplify effect with clip control.
 * @author Kai Vehmanen
 */
class EFFECT_AMPLIFY : public EFFECT_DYNAMIC {

  double kerroin;
  int nm, num_of_clipped, maxnum_of_clipped;

 public:

  string name(void) { return("Amplify"); }

  void set_parameter(int param, double value);
  double get_parameter(int param);

  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_AMPLIFY* clone(void)  { return new EFFECT_AMPLIFY(*this); }

  EFFECT_AMPLIFY (void) : num_of_clipped(0) {
    add_parameter("amp-%");
    add_parameter("max-clipped");
  }

  EFFECT_AMPLIFY (double multiplier_percent, int max_clipped);
  ~EFFECT_AMPLIFY (void) { }
};

/**
 * Amplify effect without clip control.
 * @author Kai Vehmanen
 */
class EFFECT_AMPLIFY_SIMPLE: public EFFECT_DYNAMIC {

  double kerroin;
  int nm;

 public:

  string name(void) { return("Simple amplify"); }

  void set_parameter(int param, double value);
  double get_parameter(int param);

  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_AMPLIFY_SIMPLE* clone(void)  { return new EFFECT_AMPLIFY_SIMPLE(*this); }
  EFFECT_AMPLIFY_SIMPLE (double multiplier_percent);
  EFFECT_AMPLIFY_SIMPLE (void) {
    add_parameter("amp-%");
  }
  ~EFFECT_AMPLIFY_SIMPLE (void) {  }
};

/**
 * Effect for analyzing sample volume.
 * @author Kai Vehmanen
 */
class EFFECT_ANALYZE : public EFFECT_ANALYSIS {

  int nm;
  unsigned long int num_of_samples[SAMPLE_BUFFER::ch_count]; // number of samples processed
  unsigned long int ranges[16][SAMPLE_BUFFER::ch_count];     // tyyliin ... -32768->30720 ... 30719->32767
  double max;
  
 public:

  double max_multiplier(void) const;
    
  string name(void) { return("Volume-analyze"); }
    
  void process(SAMPLE_BUFFER::sample_type *insample);
  string status(void);
  
  EFFECT_ANALYZE* clone(void)  { return new EFFECT_ANALYZE(*this); }
  EFFECT_ANALYZE (const EFFECT_ANALYZE& x);
  EFFECT_ANALYZE (void);
  ~EFFECT_ANALYZE (void);
};

/**
 * Dynamic compressor.
 * @author Kai Vehmanen
 */
class EFFECT_COMPRESS : public EFFECT_DYNAMIC {

  double crate;
  double threshold;

  unsigned short int nm;  // temp value (channel)
  double delta, new_value;
  bool first_time;
  SAMPLE_BUFFER::single_type s[SAMPLE_BUFFER::ch_count];
  SAMPLE_BUFFER::sample_type temp;     // temporary values
  SAMPLE_BUFFER::sample_type lastin, lastout;

 public:

  string name(void) { return("Compress"); }

  void set_parameter(int param, double value);
  double get_parameter(int param);

  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_COMPRESS* clone(void)  { return new EFFECT_COMPRESS(*this); }
  EFFECT_COMPRESS (const EFFECT_COMPRESS& x);
  EFFECT_COMPRESS (double compress_rate, double thold);
  EFFECT_COMPRESS (void) : first_time(true) {
    add_parameter("compression_rate");
    add_parameter("threshold");
  }
};

/**
 * Calculates DC-offset.
 * @author Kai Vehmanen
 */
class EFFECT_DCFIND : public EFFECT_ANALYSIS {

private:

  double pos_sum[SAMPLE_BUFFER::ch_count], neg_sum[SAMPLE_BUFFER::ch_count];
  double num_of_samples[SAMPLE_BUFFER::ch_count];

  SAMPLE_BUFFER::single_type tempval;
  
  unsigned short int nm;  // temp value (channel)

public:

  double get_deltafix(int channel);
    
  string name(void) { return("DC-Find"); }

  string status(void);
  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_DCFIND* clone(void)  { return new EFFECT_DCFIND(*this); }
  EFFECT_DCFIND (const EFFECT_DCFIND& x);
  EFFECT_DCFIND::EFFECT_DCFIND (void);
};

/**
 * Adjusts DC-offset.
 * @author Kai Vehmanen
 */
class EFFECT_DCFIX : public EFFECT_BASE {

private:

  SAMPLE_BUFFER::single_type deltafix[SAMPLE_BUFFER::ch_count];
  unsigned short int nm;  // temp value (channel)

public:

  string name(void) { return("DC-Fix"); }

  void set_parameter(int param, double value);
  double get_parameter(int param);

  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_DCFIX* clone(void)  { return new EFFECT_DCFIX(*this); }
  EFFECT_DCFIX (const EFFECT_DCFIX& x);
  EFFECT_DCFIX (double delta_left, double delta_right);
  EFFECT_DCFIX (void) {
    add_parameter("delta-value-left");
    add_parameter("delta-value-right");
  }
};

/**
 * Noise gate. A mono-summed is used to control the gate.
 * @author Kai Vehmanen
 */
class EFFECT_NOISEGATE_MONO : public EFFECT_DYNAMIC {

  double th_level;
  double th_time;
  double atime, htime, rtime;
  
  double th_time_lask;
  double attack_lask;
  double hold_lask;
  double release_lask;
  
  SAMPLE_BUFFER::sample_type temp_sample;
  
  double kerroin;
  
  unsigned short int nm;
  
  enum { ng_waiting, ng_attacking, ng_active, ng_holding, ng_releasing } ng_status;
  
 public:
  
  string name(void) { return("Noisegate mono"); }

  void set_parameter(int param, double value);
  double get_parameter(int param);
  
  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_NOISEGATE_MONO* clone(void)  { return new EFFECT_NOISEGATE_MONO(*this); }
  EFFECT_NOISEGATE_MONO (double thlevel_percent, double thtime, double atime, double htime, double rtime);
  EFFECT_NOISEGATE_MONO (void) {
    add_parameter("threshold_level");
    add_parameter("th_time");
    add_parameter("attack");
    add_parameter("hold");
    add_parameter("release");
  }
};

/**
 * Panning effect for controlling the stereo image.
 * @author Kai Vehmanen
 */
class EFFECT_NORMAL_PAN : public EFFECT_DYNAMIC {

private:

  double right_percent_rep;
  double l_kerroin, r_kerroin;
  
public:

  string name(void) { return("Normal pan"); }
    
  void set_parameter(int param, double value);
  double get_parameter(int param);

  void process(SAMPLE_BUFFER::sample_type *insample);
    
  EFFECT_NORMAL_PAN* clone(void)  { return new EFFECT_NORMAL_PAN(*this); }
  EFFECT_NORMAL_PAN(double right_percent);
  EFFECT_NORMAL_PAN(void) {
    add_parameter("right-%");
  }
};

#endif
