#ifndef _ECA_CHAINSETUP_H
#define _ECA_CHAINSETUP_H

#include <vector>
#include <string>
#include <map>

#include "eca-chain.h"
#include "eca-chainop.h"
#include "audioio.h"
#include "samplebuffer.h"
#include "ctrl_source.h"
#include "eca-region.h"

#include "eca-resources.h"

enum EP_MM_MODE { ep_mm_auto, ep_mm_simple, ep_mm_normal, ep_mm_mthreaded };

enum {
  TYPE_NOT_OPENED =  -1,
  TYPE_FROM_EXT =     0,
  TYPE_WAVE =         1,
  TYPE_DART =         2,
  TYPE_CDR =          3,
  TYPE_OSS =          4,
  TYPE_EWF =          5,
  TYPE_ESS =          6,
  TYPE_OSSDMA =       7,
  TYPE_MP3 =          8,
  TYPE_ALSA =         9,
  TYPE_ALSAFILE =    10,
  TYPE_ALSALOOPBACK= 16,
  TYPE_AUDIOFILE =   11,
  TYPE_RAWFILE =     12,
  TYPE_STDIN =       13,
  TYPE_STDOUT =      14,
  TYPE_NULL =        15
};

class ECA_CHAINSETUP : public ECA_REGION {

  // ---
  // Setup data
  // ---
  vector<string> active_chainids;
  int chaincount;
  AIO_PARAMS active_sinfo;
  bool active_sinfo_explicit;

  map<CHAIN::aio_id_type,vector<string> > chains_assigned_to_idev, chains_assigned_to_odev;
  map<CHAIN::aio_id_type,int> number_of_chains_assigned_to_idev,number_of_chains_assigned_to_odev;

  string setup_name;
  string setup_filename;
  bool active;
  ECA_RESOURCES* ecaresources;
  vector<string> options;

  string options_general, options_inputs, options_outputs, options_chains;

  // ---
  // Setup to strings
  // ---
  void update_option_strings(void);
  string general_options_to_string(void);
  string inputs_to_string(void);
  string outputs_to_string(void);
  string chains_to_string(void);
  string audioio_to_string(AUDIO_IO_DEVICE* aiod, const string& direction);
  string chainop_to_string(CHAIN_OPERATOR* chainop, vector<GCONTROLLER*>& gctrls);
  string gcontroller_to_string(GCONTROLLER* gctrl);

  // ---
  // Setup helper functions
  // ---
  AUDIO_IO_DEVICE* new_aio_device(const string& tname, const SIMODE mode, const AIO_PARAMS& format, int buffersize);
  void add_default_chain(void);
  void update_chain_mappings(void);

 public:

  // ---
  // Setup interpretation
  // ---
  void set_defaults (void);
  void interpret_setup_nonblocking(void);
  void interpret_setup_blocking(void);
  void interpret_general_option (const string& argu);
  void interpret_region_option (const string& argu);
  void interpret_mixmode (const string& argu);
  void interpret_setup_option (const string& argu);
  void interpret_audioio_option (const string& argu, const string& argu_param);
  void interpret_chainop_option (const string& argu);
  void interpret_effect_presets (const string& argu);
  void process_singlechain_preset(const string& preset_name);
  int get_type_fromext (const string& teksti);

  // ---
  // Setup functions (except chains themselves, all additions are done
  // to all active chains)
  // ---
  void enable(void);
  void disable(void);
  bool is_enabled(void) { return(active); }

  void add_new_chains(const vector<string>& newchains);
  void remove_chain(const string& name);
  void set_active_chains(const vector<string>& chains) { active_chainids = chains; }
  void set_explicit_format(bool enabled) { active_sinfo_explicit = enabled; }

  void add_chainop(CHAIN_OPERATOR* cotmp);
  void add_gcontroller(CONTROLLER_SOURCE* csrc, int fxparam, double low, double high);
  void add_input(AUDIO_IO_DEVICE* aiod);
  void add_output(AUDIO_IO_DEVICE* aiod);
  void remove_audio_device(const string& label);
  void attach_iodev_to_active_chains(const string& filename);

  // ---
  // Setup load/save functions
  // ---
  void load_from_file(const string& filename);
  void save(void);
  void save_to_file(const string& filename);

  // ---
  // Status/info functions
  // ---
  vector<string> get_connected_chains_to_input(CHAIN::aio_id_type
					       aiod) { return(chains_assigned_to_idev[aiod]); }
  vector<string> get_connected_chains_to_output(CHAIN::aio_id_type
						aiod) { return(chains_assigned_to_odev[aiod]); }
  vector<string> get_connected_chains_to_iodev(const string& filename);
  int number_of_connected_chains_to_input(CHAIN::aio_id_type aiod) {
    return(number_of_chains_assigned_to_idev[aiod]); }
  int number_of_connected_chains_to_output(CHAIN::aio_id_type aiod) {
    return(number_of_chains_assigned_to_odev[aiod]); }
  const CHAIN* get_chain_with_name(const string& name) const;

  // ---
  // Public data objects
  // ---
  vector<AUDIO_IO_DEVICE*> inputs;
  vector<AUDIO_IO_DEVICE*> outputs;
  vector<CHAIN*> chains;
  int buffersize;        // Buffersize in samples.
  bool raisepriority;    // Should engine raise its process priority?
  bool double_buffering; // Whether to use double-buffering on audio
                         // inputs that support it.
  enum EP_MM_MODE mixmode;
  SIMODE output_openmode;

  // ---
  // Chainsetup info
  // --

  const string& name(void) const { return(setup_name); }
  const string& filename(void) const { return(setup_filename); }
  bool is_valid(void) const;

  // --
  // Constructors and destructors
  // --
  ECA_CHAINSETUP(ECA_RESOURCES* ecarc, COMMAND_LINE& cline);
  ECA_CHAINSETUP(ECA_RESOURCES* ecarc, const string setup_file, bool fromfile = true);
  ~ECA_CHAINSETUP(void);
};

#endif


