// ------------------------------------------------------------------------
// eca-text.cpp: Textmode user-interface routines for ecasound.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <string>
#include <vector>
#include <algorithm>
#include <signal.h>

#include <kvutils.h>

#include "eca-controller.h"
#include "eca-session.h"
#include "eca-main.h"
#include "version.h"
#include "debug.h"
#include "textdebug.h"
#include "error.h"
#include "eca-text.h"

ECA_SESSION* global_pointer_to_ecaparams = 0; 
bool global_session_deleted = false;

int main(int argc, char *argv[])
{

  ECA_SESSION* ecaparams = 0;
  ecadebug->set_debug_level(0);

  signal(SIGTERM, signal_handler);
  signal(SIGINT, signal_handler);
  signal(SIGQUIT, signal_handler);
  //  signal(SIGABRT, signal_handler);

  try {
    COMMAND_LINE cline = COMMAND_LINE (argc, argv);

    if (cline.has("-o:stdout") ||
	cline.has("stdout") || 
	cline.has('q'))
      ecadebug->disable();

    ecadebug->msg(print_header());

    ecaparams = new ECA_SESSION(cline);
    global_pointer_to_ecaparams = ecaparams;  // used only for signal handling! 
    if (ecaparams->is_interactive()) start_iactive(ecaparams);
    else {
      ECA_PROCESSOR epros (ecaparams);
      epros.exec();
    }
  }
  catch(ECA_ERROR* e) {
    cerr << "---\nERROR: [" << e->error_section() << "] : \"" << e->error_msg() << "\"\n\n";
  }
  catch(...) {
    cerr << "---\nCaught an unknown exception!\n";
  }

  try {
    if (global_session_deleted == false) {
      global_session_deleted = true;
      if (ecaparams != 0) delete ecaparams;
    }
  }
  catch(ECA_ERROR* e) {
    cerr << "---\nERROR: [" << e->error_section() << "] : \"" << e->error_msg() << "\"\n\n";
  }
  catch(...) {
    cerr << "---\nCaught an unknown exception!\n";
  }
  //  ecaparams->~ECA_SESSION();
  ecadebug->flush();
    
  return(0); // everything ok!
}

void signal_handler(int signum) {
  cerr << "Unexpected interrupt... cleaning up.\n";
  if (global_session_deleted == false) {
    global_session_deleted = true;
    if (global_pointer_to_ecaparams != 0) delete global_pointer_to_ecaparams;
  }
  //  delete global_pointer_to_ecaparams;
  //  global_pointer_to_ecaparams->~ECAPARAMS();
  remove(ecasound_lockfile.c_str());
  exit(1);
}

string print_header(void) 
{
  MESSAGE_ITEM mout;

  mout << "****************************************************************************\n";
  mout << "*                [1mECASound " << ecasound_version << " (C) 1997-1999 Kai Vehmanen[0m              *\n";
  mout << "****************************************************************************\n";
  
  return(mout.to_string());
}

void start_iactive(ECA_SESSION* param) {
  ECA_CONTROLLER ctrl(param);

  if (ctrl.is_engine_ready()) ctrl.start_engine(true);

  string cmd;
  try {
    do {
      if (cmd.size() > 0) {
	ctrl.command(cmd);
      }
      if (cmd.size() == 0 || ctrl.prompt()) {
	//	cout << "\nCommand 'help' shows the online-help.\n";
	cout << "\necasound ('h' for help)> "; 
      }
    }
    while(getline(cin,cmd));
  }
  catch(int n) {
    if (n == ECA_QUIT) 
      ecadebug->msg(1, "(eca-text) Exiting...");
  }
  catch(ECA_ERROR* e) {
    cerr << "---\nERROR: [" << e->error_section() << "] : \"" << e->error_msg() << "\"\n\n";
  }
  catch(...) {
    cerr << "---\nCaught an unknown exception!\n";
  }
}
