#ifndef _ECA_SESSION_H
#define _ECA_SESSION_H

#include <vector>
#include <string>
#include <map>
#include <pthread.h>

#include "audioio.h"
#include "chain.h"
#include "chainop.h"
#include "samplebuffer.h"
#include "ctrl_source.h"

#include "eca-resources.h"

enum EP_STATUS { ep_status_running,
		 ep_status_stopped, 
		 ep_status_finished,
		 ep_status_notready };

#include "eca-chainsetup.h"

class ECA_SESSION {

  friend class ECA_CONTROLLER;
  friend class ECA_PROCESSOR;
  friend void *mthread_process_chains(void* params);

 private:

  ECA_RESOURCES ecaresources;

  // ---
  // Status data
  // ---
  pthread_mutex_t status_lock;
  EP_STATUS ep_status;
  long master_input_length;
  int master_input_id;
  bool active_setup_connected;

  vector<ECA_CHAINSETUP*> chainsetups;
  ECA_CHAINSETUP* active_chainsetup;  

  // ---
  // Setup interpretation
  // ---
  void set_defaults (void);
  void interpret_general_options(COMMAND_LINE& cline);
  void interpret_general_option (const string& argu);
  void interpret_chainsetup (const string& argu, const string& toinen);
  void connect_active_setup(void);
  void disconnect_active_setup(void);
  void select_master_input(void);
  void init_mutexes(void);

  // public:

  // ---
  // Function for handling chainsetups
  // ---
  void add_setup(ECA_CHAINSETUP* comline_setup);
  void add_new_setup(const string& name);
  void set_active_setup(const string& name);
  void save_active_setup(const string& name);
  void load_active_setup(const string& name);

  // ---
  // Control functions
  // ---

 private:

  void update_controller_sources(void);
  void status(const EP_STATUS);

  // ---
  // Functions for handling master input
  // ---
  void lock_master_input(void) { pthread_mutex_lock(in_locks[master_input_id]); }
  void release_master_input(void) { pthread_mutex_unlock(in_locks[master_input_id]); }
  AUDIO_IO_DEVICE* master_input(void) { return((*inputs)[master_input_id]); }

  // ---
  // Status/info functions
  // ---
  EP_STATUS status(void);

  vector<string> get_connected_chains_to_input(CHAIN::aio_id_type aiod) { return(active_chainsetup->get_connected_chains_to_input(aiod)); }
  vector<string> get_connected_chains_to_output(CHAIN::aio_id_type aiod) { return(active_chainsetup->get_connected_chains_to_output(aiod)); }
  int number_of_connected_chains_to_input(CHAIN::aio_id_type aiod)  {
    return(active_chainsetup->number_of_connected_chains_to_input(aiod)); }
  int number_of_connected_chains_to_output(CHAIN::aio_id_type aiod)  {
    return(active_chainsetup->number_of_connected_chains_to_output(aiod)); }

  bool is_slave_output(CHAIN::aio_id_type aiod);

 private:

  // ---
  // Thread locks
  // ---
  vector<pthread_mutex_t*> in_locks;
  vector<pthread_mutex_t*> chain_locks;
  vector<pthread_mutex_t*> out_locks;

  // ---
  // Data objects
  // ---
  map<int,bool> chain_ready_for_submix;
  vector<SAMPLE_BUFFER> inslots;
  vector<SAMPLE_BUFFER> outslots;

  // ---
  // Pointers to active chainsetup
  // ---
  vector<AUDIO_IO_DEVICE*>* inputs;
  vector<AUDIO_IO_DEVICE*>* outputs;
  vector<CHAIN*>* chains;

  // ---
  // Status data
  // ---
  int buffersize;        // Buffersize in samples.
  bool iactive;          // Should engine use 'cqueue'?
  bool mthreaded_io;     // Should engine use multithreaded io-routines?
  bool enable_outqueue;  // Should engine use 'outputqueue'?
  bool raisepriority;    // Should engine raise its process priority?
  bool mthreaded_use_of_session;
  bool multitrack_mode;
  bool loop_active;
  int loop_start_pos, loop_end_pos;
  bool sfx;              // Is SFX-section enabled?
  long loop_counter;
  enum EP_MM_MODE mixmode;

  // --
  // Public/const routines
  // --
 public:

  const vector<ECA_CHAINSETUP*>& get_chainsetups(void) const { return chainsetups; }
  const ECA_CHAINSETUP* get_active_chainsetup(void) const { return active_chainsetup; }
  const ECA_CHAINSETUP* get_chainsetup_with_name(const string& name) const;
  bool is_active_chainsetup_connected(void) const { return(active_setup_connected); }

  long length_in_samples(void) const;
  double length_in_seconds_exact(void) const;
  long position_in_samples(void) const;
  double position_in_seconds_exact(void) const;

  bool is_interactive(void) { return iactive; }

 public:

  // --
  // Constructors and destructors
  // --
  ECA_SESSION(void);
  ECA_SESSION(COMMAND_LINE& cline);
  ~ECA_SESSION(void);

 private:

  // --
  // Make sure that objects of this class aren't copy constucted/assigned
  // --
  ECA_SESSION (const ECA_SESSION& x) { }
  ECA_SESSION& operator=(const ECA_SESSION& x) { }

};

#endif


