// ------------------------------------------------------------------------
// audioio-raw.cpp: Raw/headerless audio file format input/output
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <cmath>
#include <string>
#include <cstring>
#include <cstdio>
#include <cassert>

#include "samplebuffer.h"
#include "audioio.h"
#include "audioio-raw.h"

#include "error.h"
#include "debug.h"

RAWFILE::RAWFILE(const string& name, const SIMODE mode, const AIO_PARAMS& fmt, int bsize) 
  :  AUDIO_IO_DEVICE(name, mode, fmt, bsize) {

  switch(mode) {
  case si_read:
    {
      if (name.at(0) == '-') fobject = stdin;
      else {
	fobject = fopen(name.c_str(),"rb");
	if (!fobject)
	  throw(new ECA_ERROR("AUDIOIO-RAW","Can't open " + name + " for reading."));
      }
      break;
    }
  case si_write: 
    {
      if (name.at(0) == '-') {
	cerr << "(audioio-raw) Outputting to standard output.\n";
	fobject = stdout;
      }
      else {
	fobject=fopen(name.c_str(),"wb");
	if (!fobject) 
	  throw(new ECA_ERROR("AUDIOIO-RAW","Can't open " + name + " for writing."));
      }
      break;
    }
  case si_readwrite: 
    {
      if (name.at(0) == '-') {
	cerr << "(audioio-raw) Outputting to standard output.\n";
	fobject = stdout;
      }
      else {
	fobject=fopen(name.c_str(),"r+b");
	if (!fobject) {
	  fobject=fopen(name.c_str(),"w+b");
	  if (!fobject) {	
	    throw(new ECA_ERROR("AUDIOIO-RAW","Can't open " + name + " for read-write."));
	  }
	}
      }
    }
  }
  
  set_length_in_bytes();
  first();
}

RAWFILE::~RAWFILE(void) {
  fclose(fobject);
}

void RAWFILE::get_sample(SAMPLE_BUFFER* t) {
  samples_read = fread(t->iobuf_uchar, format().align, buffersize(), fobject);
  
  if (!ferror(fobject)) {
    t->iobuf_to_buffer(format().bits, SAMPLE_BUFFER::is_system_littleendian, format().channels);
    t->length_in_samples(samples_read);
    curpos_value += samples_read * format().align;
    if (feof(fobject)) finished(true);
  }
  else
    throw(new ECA_ERROR("AUDIOIO-RAW","Read error!"));
}

void RAWFILE::put_sample(SAMPLE_BUFFER* t) {
  length_in_bytes_value = curpos_value += t->length_in_samples() * format().align;
  t->buffer_to_iobuf(format().bits, SAMPLE_BUFFER::is_system_littleendian, format().channels);
  fwrite(t->iobuf_uchar, format().align, t->length_in_samples(), fobject);
  if (ferror(fobject)) finished(true);
}

void RAWFILE::set_length_in_bytes(void) {
  long int save = ftell(fobject);
  fseek(fobject,0,SEEK_END);
  length_in_bytes_value = ftell(fobject);
  fseek(fobject,save,SEEK_SET);
}

void RAWFILE::position_in_samples(long pos) {
  curpos_value = pos * format().align;
  if (curpos_value <  0) curpos_value = 0;
  finished(false);
  fseek(fobject, curpos_value, SEEK_SET);
}



