// ------------------------------------------------------------------------
// eca-qtsession.cpp: Qt widget representing a ECA_SESSION object and 
//                    its state.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <qwidget.h>
#include <qapplication.h>
#include <qpushbutton.h>
#include <qlayout.h>
#include <qfont.h>
#include <qlistview.h>
#include <qstring.h>
#include <qfiledialog.h>
#include <qdir.h>
#include <qmessagebox.h>

#include "eca-session.h"
#include "eca-chainsetup.h"
#include "eca-controller.h"

#include "eca-qtchainsetup.h"

#include "qlistview-dumb.h"
#include "eca-qtsession.h"

#include "error.h"

QESession::QESession (ECA_CONTROLLER* econtrol, const ECA_SESSION*
		      esession, QWidget *parent=0, const char *name=0)
  : QWidget( parent, name ), ctrl(econtrol), ecasession(esession)
{
  setMinimumSize( 600, 200 );
  //  setMaximumSize( 1024, 768 );

  startTimer(5000);

  string caption = "qtecasound - session setup window";
  setCaption(caption.c_str());

  QBoxLayout* topLayout = new QVBoxLayout( this );
  QBoxLayout* buttons = new QHBoxLayout();

  init_chainsetuplist();
  init_buttons(buttons);

  topLayout->addLayout(buttons, 1);
  topLayout->addWidget(chainsetupview, 0, 0);

}

void QESession::timerEvent( QTimerEvent * ) {
  update_chainsetuplist();
}

void QESession::init_buttons(QBoxLayout* buttons) {
  QFont butfont ("Helvetica", 12, QFont::Normal);

  QPushButton* newb = new QPushButton( "(N)ew", this, "newb" );
  newb->setFont(butfont);
  buttons->addWidget( newb, 1, 0 );

  QPushButton* open = new QPushButton( "(O)pen", this, "open" );
  open->setFont(butfont);
  buttons->addWidget( open, 1, 0 );

  QPushButton* save = new QPushButton( "(S)ave", this, "save" );
  save->setFont(butfont);
  buttons->addWidget( save, 1, 0 );

  QPushButton* del = new QPushButton( "(D)elete", this, "del" );
  del->setFont(butfont);
  buttons->addWidget( del, 1, 0 );

  QPushButton* activate = new QPushButton( "(A)ctivate", this, "activat" );
  activate->setFont(butfont);
  buttons->addWidget( activate, 1, 0 );

  QPushButton* ffocus = new QPushButton( "(F)ocus to list", this, "ffocus" );
  ffocus->setFont(butfont);
  buttons->addWidget( ffocus, 1, 0);

  QPushButton* setup = new QPushButton( "Setu(p)", this, "setup" );
  setup->setFont(butfont);
  buttons->addWidget( setup, 2, 0 );


  QPushButton* quit = new QPushButton( "(Q)uit", this, "quit" );
  quit->setFont(butfont);
  buttons->addWidget( quit, 1, 0);

  QObject::connect( open, SIGNAL(clicked()), this, SLOT(button_open()));
  QObject::connect( save, SIGNAL(clicked()), this, SLOT(button_save()));
  QObject::connect( newb, SIGNAL(clicked()), this, SLOT(button_new()));
  QObject::connect( del, SIGNAL(clicked()), this, SLOT(button_del()));
  QObject::connect( activate, SIGNAL(clicked()), this, SLOT(button_activate()));
  QObject::connect( setup, SIGNAL(clicked()), this, SLOT(button_open_chainsetup()));
  QObject::connect( ffocus, SIGNAL(clicked()), chainsetupview, SLOT(setFocus()));
  QObject::connect( quit, SIGNAL(clicked()), this, SLOT(close()));
  //  connect(rewind, SIGNAL(clicked()), this, SLOT(emsg_rewind()) );
  //  connect(quit, SIGNAL(clicked()), this, SLOT(emsg_quit()) );
}

void QESession::init_chainsetuplist (void) {

  chainsetupview = new QListView_dumb(this, "chainsetupview");

  chainsetupview->setMinimumSize( 600, 100 );

  chainsetupview->addColumn("Chainsetup");
  chainsetupview->addColumn("Inputs");
  chainsetupview->addColumn("Outputs");
  chainsetupview->addColumn("Chains");
  chainsetupview->addColumn("Active");

//    chainsetupview->setColumnAlignment(0, AlignLeft);
//    chainsetupview->setColumnAlignment(1, AlignRight);
//    chainsetupview->setColumnAlignment(2, AlignRight);
//    chainsetupview->setColumnAlignment(3, AlignRight);
//    chainsetupview->setColumnAlignment(4, AlignRight);

  chainsetupview->setAllColumnsShowFocus( TRUE ); 

  update_chainsetuplist();

  int pixelsleft = width() - 5;
  for(int n = 1; n < 5; n++) {
    pixelsleft -= chainsetupview->columnWidth(n);
  }

  if (pixelsleft > 0) {
    chainsetupview->setColumnWidthMode(0, QListView::Manual);
    chainsetupview->setColumnWidth(0, pixelsleft);
  }
  chainsetupview->setGeometry(0, 0, width(), 100);

  chainsetupview->show();
}

void QESession::update_chainsetuplist (void) {
  QListViewItem* selected = chainsetupview->selectedItem();
  QString selname = ""; 
  if (selected != 0) selname = selected->text(0);
  chainsetupview->clear();

  vector<ECA_CHAINSETUP*>::const_iterator p = ecasession->get_chainsetups().begin();
  while(p != ecasession->get_chainsetups().end()) {
    //    cerr << "Adding a new one!\n";
    QString astring;
    if (ecasession->get_active_chainsetup() != 0 &&
	(*p)->name() == ecasession->get_active_chainsetup()->name())
      astring = "yes";
    else
      astring = "no";

    QListViewItem* newitem = new QListViewItem(chainsetupview,
					       (*p)->name().c_str(),
					       kvu_numtostr((*p)->inputs.size()).c_str(),
					       kvu_numtostr((*p)->outputs.size()).c_str(),
					       kvu_numtostr((*p)->chains.size()).c_str(),
					       astring);;
    if (newitem->text(0) == selname) chainsetupview->setSelected(newitem, true);
    ++p;
  }
  chainsetupview->triggerUpdate();
}

void QESession::button_open(void) {

  QFileDialog* fdialog = new QFileDialog();
  if (current_dir.exists()) fdialog->setDir(current_dir);
  QString filename = fdialog->getOpenFileName();
  if (!filename.isEmpty()) {
    try {
      ctrl->load_chainsetup(filename.latin1(), false);
    }
    catch(ECA_ERROR* e) {

      if (e->error_action() != ECA_ERROR::stop) {
	QMessageBox* mbox = new QMessageBox(this, "mbox");

	QString errormsg = "";
	errormsg += "Error while loading chainsetup: \"";
	errormsg += e->error_section().c_str();
	errormsg += "\"; ";
	errormsg += e->error_msg().c_str();
	mbox->information(this, "qtecasound", errormsg,0);
      }
    }
    //    catch(...) { throw; }

    update_chainsetuplist();
  }
}

void QESession::button_save(void) {

  QFileDialog* fdialog = new QFileDialog();
  if (current_dir.exists()) fdialog->setDir(current_dir);
  QString filename = fdialog->getSaveFileName();
  if (!filename.isEmpty()) {
    try {
      ctrl->save_chainsetup(filename.latin1(), false);
    }
    catch(ECA_ERROR* e) {
      if (e->error_action() != ECA_ERROR::stop) {
	QMessageBox* mbox = new QMessageBox(this, "mbox");

	QString errormsg = "";
	errormsg += "Error while saving chainsetup: \"";
	errormsg += e->error_section().c_str();
	errormsg += "\"; ";
	errormsg += e->error_msg().c_str();
	mbox->information(this, "qtecasound", errormsg,0);
      }
    }
  }
}

void QESession::button_new(void) {
  ctrl->new_chainsetup("untitled");
  update_chainsetuplist();
}

void QESession::button_del(void) {
  QListViewItem* item = chainsetupview->selectedItem();
  if (item != 0) {
    ctrl->delete_chainsetup(item->text(0).latin1());
    update_chainsetuplist();
  }
}

void QESession::button_activate(void) { 
  QListViewItem* item = chainsetupview->selectedItem();
  if (item != 0) {
    try {
      ctrl->activate_chainsetup(item->text(0).latin1(), false);
      update_chainsetuplist();
    }
    catch(ECA_ERROR* e) {
      if (e->error_action() != ECA_ERROR::stop) {
	QMessageBox* mbox = new QMessageBox(this, "mbox");

	QString errormsg = "";
	errormsg += "Error while activating chainsetup: \"";
	errormsg += e->error_section().c_str();
	errormsg += "\"; ";
	errormsg += e->error_msg().c_str();
	mbox->information(this, "qtecasound", errormsg,0);
      }
    }
  }
}

void QESession::button_open_chainsetup(void) { 
  QListViewItem* item = chainsetupview->selectedItem();
  if (item != 0) {
    if (ecasession->get_active_chainsetup() == 0) return;	
    if (item->text(0) !=
	QString(ecasession->get_active_chainsetup()->name().c_str()))
      {
	QMessageBox* mbox = new QMessageBox(this, "mbox");
	mbox->information(this, "qtecasound", "Only active chainsetups can be accessed!",0);
      }
    else {
      QEChainsetup* csetup = new QEChainsetup(ctrl, ecasession->get_chainsetup_with_name(item->text(0).latin1()));
      csetup->show();
    }
  }
}

void QESession::keyPressEvent(QKeyEvent* kevent) {
  switch(tolower(kevent->ascii())) {
  case 'a': 
    {    
      button_activate();
      break;
    }

  case 'd': 
    {    
      button_del();
      break;
    }

  case 'f': 
    {
      // focus to chains
      chainsetupview->setFocus();
      break;
    }

  case 'n':
    {    
      button_new();
      break;
    }

  case 'o': 
    {    
      button_open();
      break;
    }

  case 'p': 
    {    
      button_open_chainsetup();
      break;
    }

  case 'q': 
    {
      close();
      break;
    }

  case 's': 
    {    
      button_save();
      break;
    }
  }
  kevent->ignore();
}

