// ------------------------------------------------------------------------
// eca-chainsetup.cpp: A class representing a group of chains and their
//                     configuration.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <config.h>

#include <string>
#include <cstring>
#include <algorithm>
#include <vector>

#include <kvutils.h>

#include "eca-resources.h"
#include "eca-session.h"

#include "chain.h"
#include "audiofx.h"
#include "audiogate.h"
#include "audiofx_compressor.h"

#include "audioio.h"
#include "audioio-cdr.h"
#include "audioio-wave.h"
#include "audioio-oss.h"
// #include "audioio-oss_dma.h"
#include "audioio-ewf.h"
#include "ecasynth.h"
#include "audioio-mp3.h"
#include "audioio-alsa.h"
#include "audioio-af.h"
#include "audioio-raw.h"

#include "osc-gen.h"
#include "osc-sine.h"
#include "midi-cc.h"

#include "error.h"
#include "debug.h"
#include "eca-chainsetup.h"

ECA_CHAINSETUP::ECA_CHAINSETUP(ECA_RESOURCES* ecarc, COMMAND_LINE& cline) {
  setup_name = "command-line-setup";
  ecaresources = ecarc;
  buffersize = atoi(ecaresources->resource("default-buffersize").c_str());
  active_sinfo_explicit = false;
  active = false;

  cline.back_to_start();
  cline.next(); // skip program name
  string temp;
  while(cline.ready()) {
    temp = cline.next();
    if (temp == "") continue;
    ecadebug->msg(5, "(eca-chainseup) Adding \"" + temp + "\" to options.");
    options.push_back(temp);
  }
}

ECA_CHAINSETUP::ECA_CHAINSETUP(ECA_RESOURCES* ecarc, const string
			       setup_file, bool fromfile) {
  setup_name = "";
  ecaresources = ecarc;
  active = false;
  buffersize = atoi(ecaresources->resource("default-buffersize").c_str());

  if (fromfile) load_from_file(setup_file);
  if (setup_name == "") setup_name = setup_file;
}

ECA_CHAINSETUP::~ECA_CHAINSETUP(void) {
  ecadebug->msg(1,"ECA_CHAINSETUP destructor!");
  disable();
}

void ECA_CHAINSETUP::enable(void) {
  if (active == false) {
    interpret_setup();
  }
  active = true;
}

void ECA_CHAINSETUP::disable(void) {
  if (active) {
    ecadebug->control_flow("Closing chainsetup \"" + name() + "\"");

    for(vector<AUDIO_IO_DEVICE*>::iterator q = inputs.begin(); q != inputs.end(); q++) {
      ecadebug->msg("(eca-chainsetup) Closing audio device/file \"" + (*q)->label() + "\".");
      delete *q;
    }
    inputs.resize(0);
    
    for(vector<AUDIO_IO_DEVICE*>::iterator q = outputs.begin(); q != outputs.end(); q++) {
      ecadebug->msg("(eca-chainsetup) Closing audio device/file \"" + (*q)->label() + "\".");
      delete *q;
    }
    outputs.resize(0);
    
    for(vector<CHAIN*>::iterator q = chains.begin(); q != chains.end(); q++) {
      ecadebug->msg("(eca-chainsetup) Closing chain \"" + (*q)->cname() + "\".");
      delete *q;
    }
    chains.resize(0);
    active = false;
  }
}


bool ECA_CHAINSETUP::is_valid(void) {
  if (inputs.size() == 0) return(false);
  if (outputs.size() == 0) return(false);
  if (chains.size() == 0) return(false);

  return(true);
}

void ECA_CHAINSETUP::interpret_setup(void) {
  vector<string>::const_iterator p = options.begin();
  while(p != options.end()) {
    interpret_general_option(*p);
    interpret_mixmode(*p);
    ecadebug->msg(5, "(eca-chainsetup) Intepreting general option \""
		  + *p + "\".");
    ++p;
  }

  string temp, another;
  p = options.begin();
  while(p != options.end()) {
    temp = *p;
    ++p;
    if (p == options.end()) {
      another = "";
      --p;
    }
    else {
      another = *p;
      if (another != "" && another[0] == '-') {
	--p;
	another = "";
      }
    }
    ecadebug->msg(5, "(eca-chainsetup) Intepreting setup, with args \""
		  + temp + "\", \"" + another + "\".");
    interpret_setup_option(temp);
    interpret_audioio_option(temp, another);
    interpret_chainop_option(temp);
    ++p;
  }

  if (inputs.size() == 0) {
    // No -i[:] options specified; let's try to be artificially intelligent 
    p = options.begin();
    while(p != options.end()) {
      if ((*p) != "" && (*p)[0] != '-') {
	interpret_audioio_option("-i", *p);
	break;
      }
      ++p;
    }
  }
  if (inputs.size() > 0 && outputs.size() == 0) {
    // No -o[:] options specified; let's use the default output 
    vector<string> temp;
    temp.push_back("all");
    set_active_chains(temp);
    interpret_audioio_option("-o", ecaresources->resource("default-output"));
  }
}

void ECA_CHAINSETUP::interpret_general_option (const string& argu) {
  if (argu.size() < 2) return;
  switch(argu[1]) {
  case 'b':
    {
      buffersize =  atoi(get_argument_number(1, argu).c_str());
      MESSAGE_ITEM mitemb;
      mitemb << "(eca-chainsetup) Setting buffersize to (samples) " << buffersize << ".";
      ecadebug->msg(0, mitemb.to_string()); 
      break;
    }

  case 'n':
    {
      ecadebug->msg("(eca-chainsetup) Raised-priority mode enabled.");
      setup_name = get_argument_number(1, argu);
      break;
    }

  case 'r':
    {
      ecadebug->msg("(eca-chainsetup) Raised-priority mode enabled.");
      raisepriority = true;
      break;
    }

  default: { }
  }
}

void ECA_CHAINSETUP::interpret_mixmode (const string& argu) {
  //  if (argu.size() == 0) throw(new ECA_ERROR("ECA-TEXT", "empty argument", retry));
  if (argu.size() < 2) return;
  if (argu[0] != '-') return;
  switch(argu[1]) {
  case 'm':      // mixmode
    {
      string temp = get_argument_number(1, argu);
      if (temp == "auto") {
	mixmode = ep_mm_auto;
	ecadebug->msg("(eca-chainsetup) Mix-mode is selected automatically.");
      }
      else if (temp == "mthreaded") {
	ecadebug->msg("(eca-chainsetup) Multithreaded mixmode selected.");
	mixmode = ep_mm_mthreaded;
      }
      else if (temp == "simple") {
	ecadebug->msg("(eca-chainsetup) Simple mixmode selected.");
	mixmode = ep_mm_simple;
      }
      else if (temp == "normal") {
	ecadebug->msg("(eca-chainsetup) Simple mixmode selected.");
	mixmode = ep_mm_normal;
      }
    }
    break;

  default: { }
  }
}

void ECA_CHAINSETUP::interpret_setup_option (const string& argu) {
  if (argu.size() < 2) return;  
  if (argu[0] != '-') return;
  switch(argu[1]) {
  case 'a':
    {
      active_chainids = get_arguments(argu);
      add_new_chains(active_chainids);
      MESSAGE_ITEM mtempa;
      mtempa << "(eca-chainsetup) Active chain ids: ";
      for (vector<string>::const_iterator p = active_chainids.begin(); p != active_chainids.end(); p++) {
	mtempa << *p << " ";
      }
      ecadebug->msg(1, mtempa.to_string());
      break;
    }
  case 'f':
    {
      active_sinfo.bits = (unsigned short int) atoi(get_argument_number(1, argu).c_str());
      active_sinfo.channels = (unsigned short int) atoi(get_argument_number(2, argu).c_str());
      active_sinfo.srate = (unsigned short int) atoi(get_argument_number(3, argu).c_str());
      active_sinfo_explicit = true;
      
      MESSAGE_ITEM ftemp;
      ftemp << "(eca-chainsetup) Set active format to (bits/channels/srate): ";
      ftemp << (int)active_sinfo.bits << "/" << (int)active_sinfo.channels << "/" << active_sinfo.srate;
      ecadebug->msg(ftemp.to_string());
      break;
    }
  case 'p':
    {
      if (argu.size() < 3) return;  
      switch(argu[2]) {
      case 'm':
	{
	  break;
	}

      case 's': 
	{
	  process_singlechain_preset(get_argument_number(1,argu));
	  break;
	}
	
      default: { }
      }
      break;
    }
  default: { }
  }
}

void ECA_CHAINSETUP::process_singlechain_preset(const string&  preset_name) {
  ecadebug->msg(1,"(eca-chainsetup) Opening sc-preset file.");
  string filename =
    ecaresources->resource("resource-directory") + "/" + ecaresources->resource("resource-file-single-effect-presets");
  ifstream fin (filename.c_str());

  if (!fin) {
    throw(new ECA_ERROR("ECA_CHAINSETUP", "Unable to open single-chain preset file: " + filename + "."));
    return;
  }
  
  ecadebug->msg(1,"(eca-chainsetup) Starting to process sc-preset file. Trying to find \"" + preset_name + "\".");
  string sana;
  while(fin >> sana) {
    if (sana.size() > 0 && sana[0] == '#') {
      while(fin.get() != '\n' && fin.eof() == false);
      continue;
    }
    else {
      ecadebug->msg(5, "(eca-chainsetup) Next sc-preset is " + sana + ".");
      if (preset_name == sana) {
	ecadebug->msg(5, "(eca-chainsetup) Found the right preset!");

	getline(fin,sana);
	vector<string> chainops = string_to_words(sana);
	vector<string>::const_iterator p = chainops.begin();
	while (p != chainops.end()) {
	  ecadebug->msg(5, "(eca-chainsetup) Adding chainop " + *p + ".");
	  interpret_chainop_option(*p);
	  ++p;
	}
	break;
      }
      else 
	while(fin.get() != '\n' && fin.eof() == false);
    }
  }

  fin.close();
}

void ECA_CHAINSETUP::interpret_audioio_option (const string& argu, const string& argu_param) {
  if (argu.size() == 0) return;
  
  string tname = get_argument_number(1, argu);
  if (tname == "") tname = argu_param;                    // -i and -o
  else if (tname[0] == '-') tname = argu_param;           // -i and -o 
  else {                                                  // -i:
	string temp = tname;
	to_lowercase(temp);
	if (temp == "alsa") {                             // -i:alsa,c,d
	  string::const_iterator p = argu.begin();        // -o:alsa,c,d
	  ++p; ++p; ++p;
	  tname = string(p, argu.end());
	}
  }
  ecadebug->msg(5,"eca-chainsetup) adding file \"" + tname + "\".");
  if (argu.size() < 2) return;  
  if (argu[0] != '-') return;
  switch(argu[1]) {
  case 'i':
    {
      ecadebug->control_flow("Main program/Adding a new input");
      AUDIO_IO_DEVICE* aiod;
      if (active_sinfo_explicit == true) {
	aiod = new_aio_device(tname, si_read, active_sinfo, buffersize);
      }
      else {
	AIO_PARAMS default_sinfo;
	default_sinfo.bits = 16;
	default_sinfo.srate = 44100;
	default_sinfo.channels = 2;
	default_sinfo.byte_second = 176400;

	aiod = new_aio_device(tname, si_read, default_sinfo, buffersize);
      }
	
      add_input(aiod);
	
      //      MESSAGE_ITEM itemp;
      //      itemp << "Infile (" << inputs.size() - 1 << ") \"" << inputs.back()->label() << "\".";
      //      ecadebug->msg(itemp.to_string());
      break;
    }

  case 'o':
    {
      ecadebug->control_flow("Main program/Adding a new output");
      AUDIO_IO_DEVICE* aiod;
      if (active_sinfo_explicit == true)
	aiod = new_aio_device(tname, si_write, active_sinfo, buffersize);
      else {
	AIO_PARAMS default_sinfo;
	default_sinfo.bits = 16;
	default_sinfo.srate = 44100;
	default_sinfo.channels = 2;
	default_sinfo.byte_second = 176400;
	aiod = new_aio_device(tname, si_write, default_sinfo, buffersize);
      }
	
      add_output(aiod);
	
      //      MESSAGE_ITEM otemp;
      //      otemp << "Outfile (" << outputs.size() - 1 << ") \"" << outputs.back()->label() << "\".";
      //      ecadebug->msg(otemp.to_string());
      break;
    }

  default: { }
  }
}

int ECA_CHAINSETUP::get_type_fromext (const string& filename) {
  int typevar;
  string teksti = filename;
  to_lowercase(teksti);

  if (strstr(teksti.c_str(),".wav") != 0) typevar = TYPE_WAVE;
  else if (strstr(teksti.c_str(),".cdr") != 0) typevar = TYPE_CDR;
  else if (strstr(teksti.c_str(),".ewf") != 0) typevar = TYPE_EWF;
  else if (strstr(teksti.c_str(),"/dev/dsp") != 0) typevar = TYPE_OSS;
  else if (strstr(teksti.c_str(),".ess") != 0) typevar = TYPE_ESS;
  else if (strstr(teksti.c_str(),".mp3") != 0) typevar = TYPE_MP3;
  else if (strstr(teksti.c_str(),".mp2") != 0) typevar = TYPE_MP3;
  else if (strstr(teksti.c_str(),"alsa") != 0) typevar = TYPE_ALSA;
  else if (strstr(teksti.c_str(),"/dev/snd/pcm") != 0) typevar = TYPE_ALSAFILE;
  else if (strstr(teksti.c_str(),".aif") != 0) typevar = TYPE_AUDIOFILE;
  else if (strstr(teksti.c_str(),".au") != 0) typevar = TYPE_AUDIOFILE;
  else if (strstr(teksti.c_str(),".snd") != 0) typevar = TYPE_AUDIOFILE;
  else if (strstr(teksti.c_str(),".raw") != 0) typevar = TYPE_RAWFILE;
  else if (teksti == "stdin") typevar = TYPE_STDIN;
  else if (teksti == "stdout") typevar = TYPE_STDOUT;
  else typevar = TYPE_FROM_EXT;

  return(typevar);
}

AUDIO_IO_DEVICE* ECA_CHAINSETUP::new_aio_device(const string& argu, const SIMODE mode, const AIO_PARAMS& format, int buffersize) {

  if (argu.size() == 0) {
    throw(new ECA_ERROR("ECA-CHAINSETUP","Tried to create a audio device with null filename."));
  }
  string tname = get_argument_number(1, argu);

  AUDIO_IO_DEVICE* main_file;
  int type = get_type_fromext(tname);
  switch(type) {
  case TYPE_WAVE:
    main_file = new WAVEFILE (tname, mode, format, buffersize);
    break;
    
  case TYPE_CDR:
    main_file = new CDRFILE (tname, mode, format, buffersize);
    break;
    
  case TYPE_OSS:
#ifdef COMPILE_OSS
    main_file = new OSSDEVICE (tname, mode, format, buffersize);
#endif
    break;
    
  case TYPE_EWF:
    main_file = new EWFFILE (tname, mode, format, buffersize);
    break;
    
  case TYPE_ESS:
    main_file = new ECASYNTH (tname, mode, format);
    break;
    
    //  case TYPE_OSSDMA:
    // #ifdef COMPILE_OSS
    //    main_file = new OSSDMA (tname, mode, format, buffersize);
    // #endif
    //    break;
    
  case TYPE_MP3:
    main_file = new MP3FILE (tname, mode, format, buffersize);
    break;

  case TYPE_ALSAFILE:
    {
      string cardstr,devicestr;
      string::const_iterator p = tname.begin();
      while(p != tname.end() && *p != 'C') ++p;
      ++p;
      while(p != tname.end() && isdigit(*p)) {
	cardstr += " ";
	cardstr[cardstr.size() - 1] = *p;
	++p;
      }
      while(p != tname.end() && *p != 'D') ++p;
      ++p;
      while(p != tname.end() && isdigit(*p)) {
	devicestr += " ";
	devicestr[devicestr.size() - 1] = *p;
	++p;
      }
      
      int card = atoi(cardstr.c_str());
      int device = atoi(devicestr.c_str());

#ifdef COMPILE_ALSA
      main_file = new ALSADEVICE (card, device, mode, format, buffersize);
#endif
      break;
    }

  case TYPE_ALSA:
    {
      
      int card = atoi(get_argument_number(2, argu).c_str());
      int device = atoi(get_argument_number(3, argu).c_str());

#ifdef COMPILE_ALSA
      main_file = new ALSADEVICE (card, device, mode, format, buffersize);
#endif
      break;
    }

  case TYPE_AUDIOFILE:
    {
#ifdef COMPILE_AF
      main_file = new AUDIOFILE_INTERFACE (tname, mode, format, buffersize);
#endif
      break;
    }

  case TYPE_RAWFILE:
    {
      main_file = new RAWFILE (tname, mode, format, buffersize);
      break;
    }

  case TYPE_STDIN:
    {
      main_file = new RAWFILE ("-", si_read, format, buffersize);
      break;
    }

  case TYPE_STDOUT:
    {
      main_file = new RAWFILE ("-", si_write, format, buffersize);
      break;
    }

  default: 
    {
    throw(new ECA_ERROR("ECA_CHAINSETUP", "unknown file format; unable to open file " + tname + "."));
    }
  }
  return(main_file);
}

void ECA_CHAINSETUP::interpret_chainop_option (const string& argu) {
  //  if (argu.size() == 0) throw(new ECA_ERROR("ECA-TEXT", "empty argument", retry));
  CHAIN_OPERATOR* cotmp;
  
  if (argu.size() < 2) return;
  if (argu[0] != '-') return;
  switch(argu[1]) {
  case 'e':
    ecadebug->control_flow("Main program/Adding a new chain operator");
  if (argu.size() < 3) return;
    switch(argu[2]) {
    case 'a':
      if (argu[3] == ':') {
	cotmp = new EFFECT_AMPLIFY (atof(get_argument_number(1, argu).c_str()),
				    atoi(get_argument_number(2, argu).c_str()));
      }
      else {
	if (argu.size() < 4) return;
	switch(argu[3]) {
	case 's':
	  cotmp = new EFFECT_AMPLIFY_SIMPLE (atof(get_argument_number(1, argu).c_str()));
	  break;
	}
      }
      break;
    case 'c':
      if (argu.size() < 4) return;
      if (argu[3] == ':') {
	  cotmp = new EFFECT_COMPRESS (atof(get_argument_number(1, argu).c_str()),
				       atof(get_argument_number(2, argu).c_str()));

      } else {
	switch(argu[3]) {
	case 'a':
	  cotmp = new ADVANCED_COMPRESSOR (atof(get_argument_number(1, argu).c_str()),
					   atof(get_argument_number(2, argu).c_str()),
					   atof(get_argument_number(3, argu).c_str()),
					   atof(get_argument_number(4, argu).c_str()));

	}
      }
      break;

    case 'f':
      ecadebug->msg(4, "entering filter-fx-init section");
      if (argu.size() < 4) return;
      switch(argu[3]) {
      case '1':
	cotmp = new EFFECT_RESONANT_BANDPASS (atof(get_argument_number(1, argu).c_str()),
					      atof(get_argument_number(2, argu).c_str()));
	break;

      case '3':
	cotmp = new EFFECT_RESONANT_LOWPASS (atof(get_argument_number(1, argu).c_str()),
					     atof(get_argument_number(2, argu).c_str()),
					     atof(get_argument_number(3, argu).c_str()));
	break;
      case 'b':
	cotmp = new EFFECT_BANDPASS (atof(get_argument_number(1, argu).c_str()),
				     atof(get_argument_number(2, argu).c_str()));
	break;
      case 'h':
	cotmp = new EFFECT_HIGHPASS (atof(get_argument_number(1, argu).c_str()));

	break;
      case 'i': 
	cotmp = new EFFECT_INVERSE_COMB_FILTER (atoi(get_argument_number(1, argu).c_str()),
						atof(get_argument_number(2, argu).c_str()));
	break;

      case 'l':
	cotmp = new EFFECT_LOWPASS (atof(get_argument_number(1, argu).c_str()));
	break;

      case 'r':
	cotmp = new EFFECT_BANDREJECT (atof(get_argument_number(1, argu).c_str()),
				       atof(get_argument_number(2, argu).c_str()));
	break;
      case 's':
	cotmp = new EFFECT_RESONATOR (atof(get_argument_number(1, argu).c_str()),
				      atof(get_argument_number(2, argu).c_str()));
	break;
      }
      break;
    case 'n':
      if (argu.size() < 4) return;
      switch(argu[3]) {
      case 'm':
	cotmp = new EFFECT_NOISEGATE_MONO (atof(get_argument_number(1, argu).c_str()),
					   atof(get_argument_number(2, argu).c_str()),
					   atof(get_argument_number(3, argu).c_str()),
					   atof(get_argument_number(4, argu).c_str()),
					   atof(get_argument_number(5, argu).c_str()));
	break;
      }
      break;
    case 'p':
      ecadebug->msg(4, "entering pan-fx-init section");
      if (argu.size() < 4) return;
      switch(argu[3]) {
      case '3':
	//	  cotmp = new EFFECT_3D_PAN (atof(get_argument_number(1, argu).c_str()));
	break;
      case 'p':
	cotmp = new EFFECT_NORMAL_PAN (atof(get_argument_number(1, argu).c_str()));
	break;
      }
      break;
    case 't':
      ecadebug->msg(4, "entering timebased-fx-init section");
      if (argu.size() < 4) return;
      switch(argu[3]) {
      case 'd': 	   
	cotmp = new EFFECT_DELAY (atof(get_argument_number(1, argu).c_str()),
				  atoi(get_argument_number(2, argu).c_str()),
				  atoi(get_argument_number(3, argu).c_str()),
				  atof(get_argument_number(4, argu).c_str()));
	break;
      case 'f':
	cotmp = new EFFECT_FAKE_STEREO (atof(get_argument_number(1, argu).c_str()));
	break;
      case 'r': 
	cotmp = new EFFECT_REVERB (atof(get_argument_number(1, argu).c_str()),
				   atoi(get_argument_number(2, argu).c_str()),
				   atof(get_argument_number(3, argu).c_str()));
	break;
      }
      break;
    case 'v':
      cotmp = new EFFECT_ANALYZE ();
      break;
    case 'z':
      if (argu.size() < 4) return;
      switch(argu[3]) {
      case 'f':
	cotmp = new EFFECT_DCFIND ();
	break;
	
      case 'x':
	cotmp = new EFFECT_DCFIX (atof(get_argument_number(1, argu).c_str()),
				  atof(get_argument_number(2, argu).c_str()));
	break;
      }
      break;
    default:
      throw(new ECA_ERROR("ECA_CHAINSETUP", "unknown effect parameter", ECA_ERROR::notice));
    }
      
    add_chainop(cotmp);
    
    break; 
    // --- effect-section ends
      
  case 'g':
    ecadebug->msg(1, "entering gate-init section.\n");
    ecadebug->control_flow("Main program/Adding chainop gate");
    if (argu.size() < 3) return;
    switch(argu[2]) {
    case 'c':
      cotmp = new TIME_CROP_GATE(atof(get_argument_number(1,
							  argu).c_str()),
				 atof(get_argument_number(2,
							  argu).c_str()));
      break;
    case 't':
      if (get_argument_number(3,argu) == "rms") {
	cotmp = new
	  THRESHOLD_GATE(atof(get_argument_number(1,argu).c_str()),
			 atof(get_argument_number(2, argu).c_str()),
			 true);
      }
      else {
	cotmp = new
	  THRESHOLD_GATE(atof(get_argument_number(1,argu).c_str()),
			 atof(get_argument_number(2, argu).c_str()),
			 false);
      }
      break;
    default:
      throw(new ECA_ERROR("ECA-TEXT", "Unknown gate parameter", ECA_ERROR::notice));
    }

    add_chainop(cotmp);
    break;

  case 'k':
    {

    ecadebug->control_flow("Main program/Adding controller source");

    int fpar = atoi(get_argument_number(1, argu).c_str());
    double lowr = atof(get_argument_number(2, argu).c_str());
    double highr = atof(get_argument_number(3, argu).c_str());
    double step = (double)buffersize / SAMPLE_BUFFER::sample_rate;

    if (argu.size() < 3) return;
    switch(argu[2]) {
    case 'a':
      // KTYPE_VOLUME;
      break;

    case 'f':
      {
      //	aparam->active_fx->kenve.back()->type =
      //	KTYPE_FILE_ENVELOPE;
        string genosc_conf = ecaresources->resource("resource-directory") + "/" + ecaresources->resource("resource-file-genosc-envelopes");
	add_gcontroller(new GENERIC_OSCILLATOR(step,
								atof(get_argument_number(4, argu).c_str()), atoi(get_argument_number(5, argu).c_str()), genosc_conf), fpar, lowr, highr);
      break;
      }
    case 'm':
      // KTYPE_MIDI;
	add_gcontroller(new
					 MIDI_CONTROLLER(atoi(get_argument_number(4, argu).c_str()), atoi(get_argument_number(5, argu).c_str())), fpar, lowr, highr);
      break;
    case 'o':
      // KTYPE_OSCILLATOR;
      if (argu.size() < 4) return;
      switch(argu[3]) {
      case 'q':
	// KTYPE_OSCILLATOR_SQUARE
	break;
      case 's':
	add_gcontroller(new
					 SINE_OSCILLATOR(step, atof(get_argument_number(4, argu).c_str()), atof(get_argument_number(5, argu).c_str())), fpar, lowr, highr);
	break;
      case 't':
	// KTYPE_OSKILLATOR_TRIANGLE
	break;
      }
      break;
    case 'p':
      // KTYPE_PITCH;
      break;
    default:
      throw(new ECA_ERROR("ECA_CHAINSETUP", "invalid kenve parametres", ECA_ERROR::notice));
    }
    break;
    }
    
  default: { }
  }
}

void ECA_CHAINSETUP::add_default_chain() {
  chaincount++;
  chains.push_back(new CHAIN(buffersize, chaincount));
  chains.back()->cname("default");
  ecadebug->msg(1,"add_default_chain() ");
  active_chainids.push_back("default");
}


void ECA_CHAINSETUP::add_new_chains(const vector<string>& newchains) {
  for(vector<string>::const_iterator p = newchains.begin(); p != newchains.end(); p++) {
    bool exists = false;
    if (*p == "all") continue;
    for(vector<CHAIN*>::iterator q = chains.begin(); q != chains.end(); q++) {
      if (*p == (*q)->cname()) exists = true;
    }
    if (exists == false) {
      chaincount++;
      //      chains.push_back(new CHAIN(buffersize, chaincount, (*p)));
      //      chains.push_back(new CHAIN(buffersize, chaincount, (*p)));
      chains.push_back(new CHAIN(buffersize, chaincount));
      chains.back()->cname(*p);
      ecadebug->msg(1,"add_new_chains() added chain " + *p);
    }
  }
}

void ECA_CHAINSETUP::add_gcontroller(CONTROLLER_SOURCE* csrc, int fxparam, double low, double high) {
  GCONTROLLER* gtmp;
  for(vector<string>::const_iterator a = active_chainids.begin(); a != active_chainids.end(); a++) {
    for(vector<CHAIN*>::iterator q = chains.begin(); q != chains.end(); q++) {
      if (*a == (*q)->cname() || *a == "all") {
	if ((*q)->chainops.size() == 0) continue;
	if ((*q)->chainops.back()->support_for_dynamic_parameters() == false) {
	  //	  throw(new ECA_ERROR("ECAPARAMS","Chainop type"));
	  ecadebug->msg("WARNING: Chainop type \"" + 
                        (*q)->chainops.back()->label() + 
			"\" doesn't yet support dynamic parameter controlling.");
	}
	gtmp = new GCONTROLLER((*q)->chainops.back(),
			       csrc,
			       (char)fxparam, low, high);
	(*q)->gcontrollers.push_back(gtmp);
      }
    }
  }
}

void ECA_CHAINSETUP::add_chainop(CHAIN_OPERATOR* cotmp) {
  if (chains.size() == 0) add_default_chain();
  for(vector<string>::const_iterator p = active_chainids.begin(); p!= active_chainids.end(); p++) {
    //    if (active_chainids.size() > 1) 
    //      ecadebug->msg("(eca-chainsetup) Warning! Adding chainops to multiple chains is not currently supported!");
    for(vector<CHAIN*>::iterator q = chains.begin(); q != chains.end(); q++) {
      if (*p == (*q)->cname() || *p == "all") {
	ecadebug->msg(1, "Adding chainop to chain " + (*q)->cname() + ".");
	(*q)->chainops.push_back(cotmp->clone());
	//	(*q)->chainops.push_back(cotmp);
      }
    }
  }
}

void ECA_CHAINSETUP::add_input(AUDIO_IO_DEVICE* aiod) {
  if (chains.size() == 0) add_default_chain();
  inputs.push_back(aiod);

  string temp = "(eca-chainsetup) Opening file \"" + aiod->label();
  temp += "\" for ";
  if (aiod->io_mode() == si_read) temp += "reading";
  if (aiod->io_mode() == si_write) temp += "writing";
  if (aiod->io_mode() == si_readwrite) temp += "reading and writing";
  temp += ".\n";
  temp += aiod->format_info();
  temp += "\n(eca-chainsetup) Assigning file to chains:";
  for(vector<string>::const_iterator p = active_chainids.begin(); p!= active_chainids.end(); p++) {
    if (*p == "all") temp += " all";
    for(vector<CHAIN*>::iterator q = chains.begin(); q != chains.end(); q++) {
      if (*p == (*q)->cname() || *p == "all") {
	(*q)->inputid = (CHAIN::aio_id_type)(inputs.size() - 1);
	chains_assigned_to_idev[(*q)->inputid].push_back((*q)->cname());
	if (*p == "all")
	  number_of_chains_assigned_to_idev[(*q)->inputid] = (int)chains.size();
	else {
	  number_of_chains_assigned_to_idev[(*q)->inputid] =
	    (int)active_chainids.size();
	  temp += " " + *p;
	}
      }
    }
  }
  ecadebug->msg(temp);
}

void ECA_CHAINSETUP::add_output(AUDIO_IO_DEVICE* aiod) {
  if (chains.size() == 0) add_default_chain();
  string temp = "(audio-io) Opening file \"" + aiod->label();
  temp += "\" for ";
  if (aiod->io_mode() == si_read) temp += "reading";
  if (aiod->io_mode() == si_write) temp += "writing";
  if (aiod->io_mode() == si_readwrite) temp += "reading and writing";
  temp += ".\n";
  temp += aiod->format_info();

  outputs.push_back(aiod);

  temp += "\n(eca-chainsetup) Assigning file to chains:";
  for(vector<string>::const_iterator p = active_chainids.begin(); p!= active_chainids.end(); p++) {
    if (*p == "all") temp += " all";
    for(vector<CHAIN*>::iterator q = chains.begin(); q != chains.end(); q++) {
      if (*p == (*q)->cname() || *p == "all") {
	(*q)->outputid = (CHAIN::aio_id_type)(outputs.size() - 1);
	chains_assigned_to_odev[(*q)->outputid].push_back((*q)->cname());
	if (*p == "all")
	  number_of_chains_assigned_to_odev[(*q)->outputid] = (int)chains.size();
	else {
	  number_of_chains_assigned_to_odev[(*q)->outputid] =
	    (int)active_chainids.size();
	  temp += " " + *p;
	}
      }
    }
  }
  ecadebug->msg(temp);
}

void ECA_CHAINSETUP::load_from_file(const string filename) { 
  ifstream fin (filename.c_str());
  if (!fin) throw(new ECA_ERROR("ECA_CHAINSETUP", "Couldn't open setup read file: \"" + filename + "\".", ECA_ERROR::retry));

  string temp, another;
  
  while(fin >> temp) {
    ecadebug->msg(5, "(eca-chainseup) Adding \"" + temp + "\" to options (loaded from \"" + filename + "\".");
    options.push_back(temp);
  }
  fin.close();
}

void ECA_CHAINSETUP::save_to_file(const string filename) {
  ofstream fout (filename.c_str());
  if (!fout) {
    cerr << "Goin to throw an exception...\n";
    throw(new ECA_ERROR("ECA_CHAINSETUP", "Couldn't open setup save file: \"" +
  			filename + "\".", ECA_ERROR::retry));
  }
  
  fout << general_options_to_string() << "\n";
  fout << inputs_to_string() << "\n";
  fout << outputs_to_string() << "\n";
  fout << chains_to_string() << "\n";

  fout.close();
}

string ECA_CHAINSETUP::general_options_to_string(void) { 
  MESSAGE_ITEM t; 

  t << "-b:" << buffersize;

  t << " -n:\"" << setup_name << "\"";

  switch(mixmode) {
  case ep_mm_simple: {
    t << " -m:simple";
    break;
  } 
  case ep_mm_normal: {
    t << " -m:normal";
    break;
  } 
  case ep_mm_mthreaded: {
    t << " -m:mthreaded";
    break;
  } 
  default: { }
  }

  if (raisepriority) t << " -r";

  return(t.to_string());
}

string ECA_CHAINSETUP::inputs_to_string(void) { 
  MESSAGE_ITEM t; 

  CHAIN::aio_id_type p = 0;
  while (p < inputs.size()) {
    t << "-a:";
    vector<string> c = get_connected_chains_to_input(p);
    vector<string>::const_iterator cp = c.begin();
    while (cp != c.end()) {
      t << *cp;
      ++cp;
      if (cp != c.end()) t << ",";
    }
    t << " ";
    t << audioio_to_string(inputs[p], "i") << "\n";
    ++p;
  }

  return(t.to_string());
}

string ECA_CHAINSETUP::outputs_to_string(void) { 
  MESSAGE_ITEM t; 

  CHAIN::aio_id_type p = 0;
  while (p < outputs.size()) {
    t << "-a:";
    vector<string> c = get_connected_chains_to_output(p);
    vector<string>::const_iterator cp = c.begin();
    while (cp != c.end()) {
      t << *cp;
      ++cp;
      if (cp != c.end()) t << ",";
    }
    t << " ";
    t << audioio_to_string(outputs[p], "o") << "\n";
    ++p;
  }

  return(t.to_string());
}

string ECA_CHAINSETUP::audioio_to_string(AUDIO_IO_DEVICE* aiod, const string& direction) {
  MESSAGE_ITEM t;

  t << "-f:" << (int)aiod->format().bits << "," <<
    (int)aiod->format().channels << ","  << (int)aiod->format().srate;
  t << " -" << direction << " ";
  t << aiod->label();

  return(t.to_string());
}

string ECA_CHAINSETUP::chains_to_string(void) { 
  MESSAGE_ITEM t; 

  CHAIN::aio_id_type p = 0;
  while (p < chains.size()) {
    t << "-a:" << chains[p]->cname() << " ";
    CHAIN::aio_id_type q = 0;
    while (q < chains[p]->chainops.size()) {
      t << chainop_to_string(chains[p]->chainops[q], chains[p]->gcontrollers) << " ";
      ++q;
    }
    t << "\n";
    ++p;
  }

  return(t.to_string());
}

string ECA_CHAINSETUP::chainop_to_string(CHAIN_OPERATOR* chainop,
					 vector<GCONTROLLER*>& gctrls) {
  MESSAGE_ITEM t;

  //  if (dynamic_cast<EFFECT_AMPLIFY_SIMPLE*> (chainop))

  t << "-" << chainop->id_string();
  if (chainop->number_of_params() > 0) t << ":";
  for(int n = 0; n < chainop->number_of_params(); n++) {
    t << chainop->get_parameter(n + 1);
    if (n + 1 < chainop->number_of_params()) t << ",";
  }

  int p = 0;
  while (p < gctrls.size()) {
    if (chainop == gctrls[p]->chainop_pointer()) {
      t << " " << gcontroller_to_string(gctrls[p]);
    }
    ++p;
  } 

  return(t.to_string());
}

string ECA_CHAINSETUP::gcontroller_to_string(GCONTROLLER* gctrl) {
  MESSAGE_ITEM t; 
  t << "-" << gctrl->ctrl_source_pointer()->id_string() << ":";
  t << (int)gctrl->param_number() << ",";
  t << (int)gctrl->low_range_limit() << ",";
  t << (int)gctrl->high_range_limit();
  if (gctrl->ctrl_source_pointer()->number_of_params() > 0) t << ",";
  for(int n = 0; n < gctrl->ctrl_source_pointer()->number_of_params(); n++) {
    t << gctrl->ctrl_source_pointer()->get_parameter(n + 1);
    if (n + 1 < gctrl->ctrl_source_pointer()->number_of_params()) t << ",";
  }

  return(t.to_string());
}

const CHAIN* ECA_CHAINSETUP::get_chain_with_name(const string& name) const {
  vector<CHAIN*>::const_iterator p = chains.begin();
  while(p != chains.end()) {
    if ((*p)->cname() == name) return(*p);
    ++p;
  }
  return(0);
}
