// ------------------------------------------------------------------------
// audioio-alsa.cpp: ALSA (/dev/snd/pcm*) input/output.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <config.h>
#ifdef COMPILE_ALSA

#include <string>
#include <cstring>
#include <cstdio>

#include <kvutils.h>

#include "samplebuffer.h"
#include "audioio.h"
#include "audioio-alsa.h"

#include "error.h"
#include "debug.h"

ALSADEVICE::ALSADEVICE (int card, int device, const SIMODE
			mode, const AIO_PARAMS& form, int bsize)
  :  AUDIO_IO_DEVICE(string("alsa,") + kvu_numtostr(card) +
		     string(",") + kvu_numtostr(device) , mode, form, bsize)
{
    card_number = card;
    device_number = device;
    //    format(fmt);
    finished(false);

    open_device();

    is_triggered = false;
}

void ALSADEVICE::open_device(void) {
  if (is_open == true) return;
  int err;
  if (io_mode() == si_read) {
#ifdef OLD_ALSALIB    
    if ((err = snd_pcm_open(&audio_fd, card_number, device_number,
			    SND_PCM_OPEN_RECORD)) < 0) {
#else
    if ((err = snd_pcm_open(&audio_fd, card_number, device_number,
			    SND_PCM_OPEN_CAPTURE)) < 0) {
#endif
      throw(new ECA_ERROR("AUDIOIO-ALSA", "unable to open ALSA-device for recording; error: " + string(snd_strerror(err))));
    }
  }    
  else if (io_mode() == si_write) {
    if ((err = snd_pcm_open(&audio_fd, card_number, device_number,
			    SND_PCM_OPEN_PLAYBACK)) < 0) {
      // Opening device failed
      throw(new ECA_ERROR("AUDIOIO-ALSA", "unable to open ALSA-device for playback; error: " +  string(snd_strerror(err))));
    }
    // output triggering
    snd_pcm_playback_pause(audio_fd, 1);
  }
  else if (io_mode() == si_readwrite) {
      throw(new ECA_ERROR("AUDIOIO-ALSA", "Simultanious intput/ouput not supported."));
  }
  
  // ---
  // Set blocking mode.
  // ---
  snd_pcm_block_mode(audio_fd, 1);    // enable block mode

  // ---
  // Set fragment size.
  // ---
  if (buffersize() == 0) 
    throw(new ECA_ERROR("AUDIOIO-ALSA", "Buffersize() is 0!", ECA_ERROR::stop));
    
  if (io_mode() == si_read) {
#ifdef OLD_ALSALIB
    snd_pcm_record_info_t pcm_info;
    snd_pcm_record_info(audio_fd, &pcm_info);
    snd_pcm_record_params_t pp;
#else
    snd_pcm_capture_info_t pcm_info;
    snd_pcm_capture_info(audio_fd, &pcm_info);
    snd_pcm_capture_params_t pp;
#endif
    memset(&pp, 0, sizeof(pp));

    if (buffersize() * format().align > pcm_info.buffer_size) 
      throw(new ECA_ERROR("AUDIOIO-ALSA", "Buffer size too big, can't setup fragments."));

    pp.fragment_size = buffersize() * format().align;
    pp.fragments_min = 1;

#ifdef OLD_ALSALIB    
    err = snd_pcm_record_params(audio_fd, &pp);
#else
    err = snd_pcm_capture_params(audio_fd, &pp);
#endif
    if (err < 0) {
      throw(new ECA_ERROR("AUDIOIO-ALSA", "Error when setting up buffer fragments: " + string(snd_strerror(err))));
    }
  }
  else {
    snd_pcm_playback_info_t pcm_info;
    snd_pcm_playback_info(audio_fd, &pcm_info);

    snd_pcm_playback_params_t pp;
    memset(&pp, 0, sizeof(pp));

    pp.fragment_size = buffersize() * format().align;
    pp.fragments_max = pcm_info.buffer_size / pp.fragment_size;
    pp.fragments_room = 1;
    
    err = snd_pcm_playback_params(audio_fd, &pp);
    if (err < 0) {
      throw(new ECA_ERROR("AUDIOIO-ALSA", "Error when setting up buffer fragments: " + string(snd_strerror(err))));
    }
  }

  // ---
  // Select audio format
  // ---

  snd_pcm_format_t pf;

  memset(&pf, 0, sizeof(pf));
  if (format().bits == 16)
    pf.format = SND_PCM_SFMT_S16_LE; // SND_PCM_FMT_U16_LE
  else 
    pf.format = SND_PCM_SFMT_U8;

  pf.rate = format().srate;
  pf.channels = format().channels; // Stereo

  if (io_mode() == si_read) {
#ifdef OLD_ALSALIB
    err = snd_pcm_record_format(audio_fd, &pf);
#else
    err = snd_pcm_capture_format(audio_fd, &pf);
#endif
    if (err < 0) {
    throw(new ECA_ERROR("AUDIOIO-ALSA", "Error when setting up record parameters: " + string(snd_strerror(err))));
    }
  }
  else {
    err = snd_pcm_playback_format(audio_fd, &pf);
    if (err < 0) {
      throw(new ECA_ERROR("AUDIOIO-ALSA", "Error when setting up playback parameters: " + string(snd_strerror(err))));
    }
  }
  is_open = true;
}

void ALSADEVICE::rt_stop(void) {
  ecadebug->msg("(audioio-alsa) Audio device \"" + label() + "\" disabled.");
  if (io_mode() == si_write) {
    snd_pcm_playback_pause(audio_fd, 1);
  }
  else {
    if (is_open) close_device();
  }
  is_triggered = false;
}

void ALSADEVICE::close_device(void) {
  if (is_open) {
    if (io_mode() == si_write) snd_pcm_drain_playback(audio_fd);
#ifdef OLD_ALSALIB
    else if (io_mode() == si_read) snd_pcm_flush_record(audio_fd);
#else
    else if (io_mode() == si_read) snd_pcm_flush_capture(audio_fd);
#endif
    snd_pcm_close(audio_fd);
  }    
  is_open = false;
}

void ALSADEVICE::rt_ready(void) {
  ecadebug->msg("(audioio-alsa) Audio device \"" + label() + "\" ready.");
  if (is_open == false) {
    open_device();
  }    
}

void ALSADEVICE::rt_activate(void) {
  if (is_triggered == false) {
    if (io_mode() == si_write) {
      snd_pcm_playback_status_t pb_status;
      snd_pcm_playback_status(audio_fd, &pb_status);
      ecadebug->msg(2, "(audioio-alsa) Bytes in output-queue: " + 
		    kvu_numtostr(pb_status.queue) + ".");
      snd_pcm_playback_pause(audio_fd, 0);
    }
    is_triggered = true;
  }
}

void ALSADEVICE::get_sample(SAMPLE_BUFFER* t) {
  if (!is_open) throw(new ECA_ERROR("AUDIOIO-ALSA","get_sample(): trying to read from a closed device!"));

  samples_read = snd_pcm_read(audio_fd, t->iobuf_uchar, format().align * t->length_in_samples());

  if (samples_read < 0)
    throw(new ECA_ERROR("AUDIOIO-ALSA","get_sample(): read error!", ECA_ERROR::stop));

  //    t->length_in_samples(samples_read / format().align);
  //  else 

  t->iobuf_to_buffer(format().bits, true, format().channels);
}

void ALSADEVICE::put_sample(SAMPLE_BUFFER* t) {
  if (!is_open) throw(new ECA_ERROR("AUDIOIO-ALSA","put_sample(): trying to write to a closed device!"));
  
  //  t->buffer_to_iobuf(format().bits, SAMPLE_BUFFER::is_system_littleendian, format().channels);

  t->buffer_to_iobuf(format().bits, true, format().channels);
  snd_pcm_write(audio_fd, t->iobuf_uchar, format().align * t->length_in_samples());
}

ALSADEVICE::~ALSADEVICE(void) { close_device(); }

#endif // COMPILE_ALSA













