// ------------------------------------------------------------------------
// audiofx.cpp: Generel effect processing routines.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <cmath>

#include <kvutils.h>

#include "samplebuffer.h"
#include "audiofx.h"
#include "debug.h"
#include "error.h"

EFFECT_AMPLIFY::EFFECT_AMPLIFY (double multiplier_percent, int max_clipped) {
  kerroin = multiplier_percent / 100.0;
  maxnum_of_clipped = max_clipped ;

  num_of_clipped = 0;
  support_for_dynamic_parameters(true);
  
  MESSAGE_ITEM otemp;
  otemp.setprecision(0);
  otemp << "(fx) Amplifying sample volume by ";
  otemp << kerroin * 100.0 << "% (will warn if number of consecutive clipped samples reaches ";
  otemp << maxnum_of_clipped << ").";
  ecadebug->msg(otemp.to_string());
}

void EFFECT_AMPLIFY::set_parameter(int param, double value) {
  switch (param) {
  case 1: 
    kerroin = value / 100.0;
    break;
  case 2:
    maxnum_of_clipped = (int)value;
    break;
  }
}

double EFFECT_AMPLIFY::get_parameter(int param) { 
  switch (param) {
  case 1: 
    return(kerroin * 100.0);
  case 2:
    return(maxnum_of_clipped);
  }
}

void EFFECT_AMPLIFY::process(SAMPLE_BUFFER::sample_type *insample) {

  insample->put_left( insample->get_left() * kerroin );
  insample->put_right( insample->get_right() * kerroin );
    
  if (insample->is_clipped() == true) num_of_clipped++;
  else num_of_clipped = 0;

  if (num_of_clipped > maxnum_of_clipped && maxnum_of_clipped != 0) {
    MESSAGE_ITEM otemp;
    otemp.setprecision(0);
    otemp << "(fx) WARNING! Signal is clipping! ";
    otemp << num_of_clipped;
    otemp << " consecutive clipped samples.";
    ecadebug->msg(otemp.to_string());
  }
}

EFFECT_AMPLIFY_SIMPLE::EFFECT_AMPLIFY_SIMPLE (double multiplier_percent) {
    kerroin = multiplier_percent / 100.0;
  
    support_for_dynamic_parameters(true);

    MESSAGE_ITEM otemp;
    otemp.setprecision(0);
    otemp << "(fx) Amplifying (simple) sample volume by "  << kerroin * 100.0 << "%.";
    ecadebug->msg(otemp.to_string());
}

void EFFECT_AMPLIFY_SIMPLE::set_parameter(int param, double value) {
  switch (param) {
  case 1: 
    kerroin = value / 100.0;
    break;
  }
}

double EFFECT_AMPLIFY_SIMPLE::get_parameter(int param) { 
  switch (param) {
  case 1: 
    return(kerroin * 100.0);
  }
}

void EFFECT_AMPLIFY_SIMPLE::process(SAMPLE_BUFFER::sample_type *insample) {
    insample->put_left( insample->get_left() * kerroin );
    insample->put_right( insample->get_right() * kerroin );
}

EFFECT_ANALYZE::EFFECT_ANALYZE (void) {
  for(int ch = 0; ch < SAMPLE_BUFFER::ch_count; ch++)
    num_of_samples[ch] = 0;
  for(nm = 0; nm < 16; nm++)
    for(int ch = 0; ch < SAMPLE_BUFFER::ch_count; ch++)
      ranges[nm][ch] = 0;
  ecadebug->msg("(fx) Analyzing sample data.");
}

EFFECT_ANALYZE::~EFFECT_ANALYZE (void) {
  string temp;
  status(&temp);
  ecadebug->msg(temp);
}

EFFECT_ANALYZE::EFFECT_ANALYZE (const EFFECT_ANALYZE& x) {
  for(int ch = 0; ch < SAMPLE_BUFFER::ch_count; ch++)
    num_of_samples[ch] = x.num_of_samples[ch];
  for(nm = 0; nm < 16; nm++)
    for(int ch = 0; ch < SAMPLE_BUFFER::ch_count; ch++)
      ranges[nm][ch] = x.ranges[nm][ch];
  nm = 0;
}

void EFFECT_ANALYZE::status(string* output) {
    *output = *output + "(fx) -- Printing volume statistics --\n";
    for(nm = 0; nm < 16; nm++) {
        MESSAGE_ITEM otemp;
        otemp.setprecision(2);
        otemp << "(fx) Vol-range: ";
        otemp << nm << "\t L: ";
        otemp << ranges[nm][SAMPLE_BUFFER::ch_left] << " (";
        otemp << ((double)ranges[nm][SAMPLE_BUFFER::ch_left] / (double)num_of_samples[SAMPLE_BUFFER::ch_left] * 100.0);
        otemp << ") \t\t";
        otemp << "R: ";
        otemp << ranges[nm][SAMPLE_BUFFER::ch_right] << " (";
        otemp << ((double)ranges[nm][SAMPLE_BUFFER::ch_right] / (double)num_of_samples[SAMPLE_BUFFER::ch_right] * 100.0);
        otemp << ").\n";
        *output = *output + otemp.to_string();
    }
    *output = *output + "(fx) -- End of statistics (counters reseted)--";
    for(nm = 0; nm < 16; nm++)
        for(int ch = 0; ch < SAMPLE_BUFFER::ch_count; ch++)
            ranges[nm][ch] = 0;
    for(int ch = 0; ch < SAMPLE_BUFFER::ch_count; ch++)
        num_of_samples[ch] = 0;
}

void EFFECT_ANALYZE::process(SAMPLE_BUFFER::sample_type *insample) {
    for(nm = 0;nm < SAMPLE_BUFFER::ch_count; nm++) {
        num_of_samples[nm]++;
        if (insample->get_channel(nm)      <  SAMPLE_BUFFER::max_amplitude * -7.0/8.0) ranges[0][nm]++;
        else if (insample->get_channel(nm) <  SAMPLE_BUFFER::max_amplitude * -6.0/8.0) ranges[1][nm]++;
        else if (insample->get_channel(nm) <  SAMPLE_BUFFER::max_amplitude * -5.0/8.0) ranges[2][nm]++;
        else if (insample->get_channel(nm) <  SAMPLE_BUFFER::max_amplitude * -4.0/8.0) ranges[3][nm]++;
        else if (insample->get_channel(nm) <  SAMPLE_BUFFER::max_amplitude * -3.0/8.0) ranges[4][nm]++;
        else if (insample->get_channel(nm) <  SAMPLE_BUFFER::max_amplitude * -2.0/8.0) ranges[5][nm]++;
        else if (insample->get_channel(nm) <  SAMPLE_BUFFER::max_amplitude * -1.0/8.0) ranges[6][nm]++;
        else if (insample->get_channel(nm) <  SAMPLE_BUFFER::silent_value) ranges[7][nm]++;
        else if (insample->get_channel(nm) <  SAMPLE_BUFFER::max_amplitude * 1.0/8.0) ranges[8][nm]++;
        else if (insample->get_channel(nm) <  SAMPLE_BUFFER::max_amplitude * 2.0/8.0) ranges[9][nm]++;
        else if (insample->get_channel(nm) <  SAMPLE_BUFFER::max_amplitude * 3.0/8.0) ranges[10][nm]++;
        else if (insample->get_channel(nm) <  SAMPLE_BUFFER::max_amplitude * 4.0/8.0) ranges[11][nm]++;
        else if (insample->get_channel(nm) <  SAMPLE_BUFFER::max_amplitude * 5.0/8.0) ranges[12][nm]++;
        else if (insample->get_channel(nm) <  SAMPLE_BUFFER::max_amplitude * 6.0/8.0) ranges[13][nm]++;
        else if (insample->get_channel(nm) <  SAMPLE_BUFFER::max_amplitude * 7.0/8.0) ranges[14][nm]++;
        else if (insample->get_channel(nm) <  SAMPLE_BUFFER::max_amplitude) ranges[15][nm]++;
    }
}

EFFECT_COMPRESS::EFFECT_COMPRESS (double compress_rate, double thold) {
  set_parameter(1, compress_rate);
  set_parameter(2, thold);

  support_for_dynamic_parameters(true);
  first_time = true;

  MESSAGE_ITEM otemp;
  otemp.setprecision(2);
  otemp << "(fx) Compression enabled, compression rate ";
  otemp << crate << ", threshold " << threshold << "; lowpass filter freq is 20.";
  ecadebug->msg(otemp.to_string());
}

EFFECT_COMPRESS::EFFECT_COMPRESS (const EFFECT_COMPRESS& x) {
  crate = x.crate;
  threshold = x.threshold;
  delta = x.delta;
  first_time = x.first_time;
  for(nm = 0;nm < SAMPLE_BUFFER::ch_count; nm++)
    s[nm] = x.s[nm];
  temp = x.temp;
  lastin = x.lastin;
  lastout = x.lastout;
}

void EFFECT_COMPRESS::set_parameter(int param, double value) {
  switch (param) {
  case 1: 
    crate = pow(2.0, value / 3.0);
    break;
  case 2: 
    threshold = value;
    break;
  }
}

double EFFECT_COMPRESS::get_parameter(int param) { 
  switch (param) {
  case 1: 
    return(crate);
  case 2: 
    return(threshold);
  }
}

void EFFECT_COMPRESS::process(SAMPLE_BUFFER::sample_type *insample) {
  //  temp.put_left(fabs(insample->get_left()));
  //  temp.put_right(fabs(insample->get_right()));

  if (first_time) {
    first_time = false;
    lastin = lastout = *insample;
  }
  else {
    for(nm = 0;nm < SAMPLE_BUFFER::ch_count; nm++) {
      // cerr << "In: " << insample->get_channel(nm) << " (ch:" << nm << ")\n";
      delta = insample->get_channel(nm) - lastin.get_channel(nm);
      delta = delta / crate;
      //      delta = (delta - 1.0) / crate;
      //      delta += 1.0;
      //      temp.put_channel(nm, insample->get_channel(nm) * delta);
      temp.put_channel(nm, lastout.get_channel(nm) + delta);
      // cerr << "Out: " << temp.get_channel(nm) << " (ch:" << nm << ")\n";
    }
    lastin = *insample;
    if (fabs(insample->get_left()) < threshold || fabs(insample->get_right()) < threshold) {
      //      cerr << "T";
      lastout = *insample;
    }
    else {
      //      cerr << "delta: " << delta << "\n";
      *insample = temp;
      lastout = temp;
    }
  }
  //  lowpass.process(&temp);
  //  insample->put_left(pow(temp.get_left(), crate) * insample->get_left());
  //  insample->put_right(pow(temp.get_right(), crate) * insample->get_right());
}

EFFECT_DCFIND::EFFECT_DCFIND (void) {
    for(nm = 0; nm < SAMPLE_BUFFER::ch_count; nm++) {
        pos_sum[nm] = neg_sum[nm] = num_of_samples[nm] = 0.0;
    }
    
    ecadebug->msg("(fx) Finding optimal value for adjusting DC.");
}

EFFECT_DCFIND::EFFECT_DCFIND (const EFFECT_DCFIND& x) {
  for(nm = 0; nm < SAMPLE_BUFFER::ch_count; nm++) {
    pos_sum[nm] = x.pos_sum[nm];
    neg_sum[nm] = x.neg_sum[nm];
    num_of_samples[nm] = x.num_of_samples[nm];
    deltafix[nm] = x.deltafix[nm];
  }
  tempval = x.tempval;
  nm = x.nm;
}

void EFFECT_DCFIND::status(string* output) {
    for(nm = 0; nm < SAMPLE_BUFFER::ch_count; nm++) {    
        if (pos_sum[nm] > neg_sum[nm]) deltafix[nm] = -(pos_sum[nm] - neg_sum[nm]) / num_of_samples[nm];
        else deltafix[nm] = (neg_sum[nm] - pos_sum[nm]) / num_of_samples[nm];
    }
    MESSAGE_ITEM mitem;
    mitem.setprecision(5);
    mitem << "(fx) Optimal value for DC-adjust: ";
    mitem << deltafix[SAMPLE_BUFFER::ch_left] << " (left), ";
    mitem << deltafix[SAMPLE_BUFFER::ch_right] << " (right).";
    *output = *output + mitem.to_string();
}

void EFFECT_DCFIND::process(SAMPLE_BUFFER::sample_type *insample) {
    for(nm = 0; nm < SAMPLE_BUFFER::ch_count; nm++) {
        tempval = insample->get_channel(nm);
        if (tempval > SAMPLE_BUFFER::silent_value)
            pos_sum[nm] += tempval;
        else
            neg_sum[nm] += fabs(tempval);
        num_of_samples[nm]++;
    }
}

EFFECT_DCFIX::EFFECT_DCFIX (double delta_left, double delta_right) {
    deltafix[SAMPLE_BUFFER::ch_left] = delta_left;
    deltafix[SAMPLE_BUFFER::ch_right] = delta_right;

    support_for_dynamic_parameters(true);

    MESSAGE_ITEM mitem;
    mitem.setprecision(5);
    mitem << "(fx) Adjusting DC with delta value: ";
    mitem << deltafix[SAMPLE_BUFFER::ch_left] << " (left), ";
    mitem << deltafix[SAMPLE_BUFFER::ch_right] << " (right).";
    ecadebug->msg(mitem.to_string());    
}

EFFECT_DCFIX::EFFECT_DCFIX (const EFFECT_DCFIX& x) {
  for(nm = 0; nm < SAMPLE_BUFFER::ch_count; nm++) {
    deltafix[nm] = x.deltafix[nm];
  }
}

void EFFECT_DCFIX::set_parameter(int param, double value) {
  switch (param) {
  case 1: 
    deltafix[SAMPLE_BUFFER::ch_left] = value;
    break;
  case 2: 
    deltafix[SAMPLE_BUFFER::ch_right] = value;
    break;
  }
}

double EFFECT_DCFIX::get_parameter(int param) { 
  switch (param) {
  case 1: 
    return(deltafix[SAMPLE_BUFFER::ch_left]);
  case 2: 
    return(deltafix[SAMPLE_BUFFER::ch_right]);
  }
}

void EFFECT_DCFIX::process(SAMPLE_BUFFER::sample_type *insample) {
    for(nm = 0; nm < SAMPLE_BUFFER::ch_count; nm++)
        insample->put_channel(nm, (insample->get_channel(nm) + deltafix[nm]));
}

EFFECT_NOISEGATE_MONO::EFFECT_NOISEGATE_MONO (double thlevel_percent, double thtime, double a, double h, double r) {
    th_level = SAMPLE_BUFFER::max_amplitude * (thlevel_percent / 100.0);
    th_time = thtime;
    atime = a;
    htime = h;
    rtime = r;

    support_for_dynamic_parameters(true);

    MESSAGE_ITEM otemp;
    otemp.setprecision(0);
    otemp << "(audiofx) Mono noise gate enabled; threshold level ";
    otemp << thlevel_percent << "%, threshold time ";
    otemp << th_time << " ms, attack ";
    otemp << atime << " ms, hold ";
    otemp << htime << " ms and release ";
    otemp << rtime << " ms.";
    ecadebug->msg(otemp.to_string());

    th_time = (htime * (double)SAMPLE_BUFFER::sample_rate * (double)SAMPLE_BUFFER::ch_count / 1000.0);
    atime = (atime * (double)SAMPLE_BUFFER::sample_rate * (double)SAMPLE_BUFFER::ch_count / 1000.0);
    htime = (atime * (double)SAMPLE_BUFFER::sample_rate * (double)SAMPLE_BUFFER::ch_count / 1000.0);
    rtime = (rtime * (double)SAMPLE_BUFFER::sample_rate * (double)SAMPLE_BUFFER::ch_count / 1000.0);

    th_time_lask = 0.0;
    attack_lask = 0.0;
    hold_lask = 0.0;
    release_lask = 0.0;

    ng_status = ng_waiting;
}

void EFFECT_NOISEGATE_MONO::set_parameter(int param, double value) {
  switch (param) {
  case 1: 
    th_level = SAMPLE_BUFFER::max_amplitude * (value / 100.0);
    break;
  case 2: 
    th_time = (value * (double)SAMPLE_BUFFER::sample_rate * (double)SAMPLE_BUFFER::ch_count / 1000.0);
    break;
  case 3: 
    th_time = (value * (double)SAMPLE_BUFFER::sample_rate * (double)SAMPLE_BUFFER::ch_count / 1000.0);
    break;
  case 4: 
    th_time = (value * (double)SAMPLE_BUFFER::sample_rate * (double)SAMPLE_BUFFER::ch_count / 1000.0);
    break;
  case 5: 
    th_time = (value * (double)SAMPLE_BUFFER::sample_rate * (double)SAMPLE_BUFFER::ch_count / 1000.0);
    break;
  }
}

double EFFECT_NOISEGATE_MONO::get_parameter(int param) { 
  switch (param) {
  case 1: 
    return(th_level * 100.0 / (double)SAMPLE_BUFFER::max_amplitude);
  case 2: 
    return(th_time * 1000.0 / (double)SAMPLE_BUFFER::sample_rate / (double)SAMPLE_BUFFER::ch_count);
  case 3: 
    return(atime * 1000.0 / (double)SAMPLE_BUFFER::sample_rate / (double)SAMPLE_BUFFER::ch_count);
  case 4: 
    return(htime * 1000.0 / (double)SAMPLE_BUFFER::sample_rate / (double)SAMPLE_BUFFER::ch_count);
  case 5: 
    return(rtime * 1000.0 / (double)SAMPLE_BUFFER::sample_rate / (double)SAMPLE_BUFFER::ch_count);
  }
}

void EFFECT_NOISEGATE_MONO::process(SAMPLE_BUFFER::sample_type *insample) {
  temp_sample = *insample;
  temp_sample.sum_to_mono();
    
  if (fabs((double)insample->get_left()) <= th_level) {
    if (ng_status == ng_holding || ng_status == ng_releasing) {
      ng_status = ng_waiting; 
      ecadebug->msg(10,"(fx) ng_mono waiting (again)");
    }

    if (ng_status == ng_active) {
      insample->make_silent();
      return;
    }

    if (ng_status == ng_attacking) {
      attack_lask++;
      kerroin = (1 - (attack_lask / atime));
      if (attack_lask >= atime) {
	ng_status = ng_active;
	ecadebug->msg(10,"(fx) ng_mono active");
	attack_lask = 0.0;
      }
    }

    if (ng_status == ng_waiting) {
      th_time_lask++;
      if (th_time_lask >= th_time) {
	ng_status = ng_attacking;
	ecadebug->msg(10,"(fx) ng_mono attacking");
	th_time_lask = 0.0;
      }
      return;
    }
  }
  else {
    if (ng_status == ng_releasing) {
      release_lask++;
      kerroin = release_lask / rtime;
      if (release_lask >= rtime) {
	ecadebug->msg(10,"(fx) ng_mono waiting");
        ng_status = ng_waiting;
	release_lask = 0.0;
      }
    }

    if (ng_status == ng_holding) {
      hold_lask++;
      if (hold_lask >= htime) {
	ng_status = ng_releasing;
	ecadebug->msg(10,"(fx) ng_mono releasing");
	hold_lask = 0.0;
      }
      insample->make_silent();
      return;
    }

    if (ng_status == ng_active) {
      ng_status = ng_holding;
      ecadebug->msg(10,"(fx) ng_mono holding");
      return;
    }

    if (ng_status == ng_waiting
	|| ng_status == ng_attacking) {
      ng_status = ng_waiting; 
      th_time_lask = 0.0;
      //      ecadebug->msg(10,"(fx) ng_mono waiting (reseted)");
      return;
    }
  }

  insample->put_left(insample->get_left() * kerroin);
  insample->put_right(insample->get_right() * kerroin);
}

EFFECT_NORMAL_PAN::EFFECT_NORMAL_PAN (double right_percent) {
    r_kerroin = right_percent / 100.0;
    l_kerroin = (1.0 - r_kerroin);

    support_for_dynamic_parameters(true);

    MESSAGE_ITEM otemp;
    otemp.setprecision(0);
    otemp << "(fx) Normal pan enabled; ";
    otemp << "left " << (l_kerroin * 100.0);
    otemp << " - right " << (r_kerroin * 100.0);
    otemp << ".";
    ecadebug->msg(otemp.to_string());
}

void EFFECT_NORMAL_PAN::set_parameter(int param, double value) {
  switch (param) {
  case 1: 
    r_kerroin = value / 100.0;
    l_kerroin = (1.0 - r_kerroin);
    break;
  }
}

double EFFECT_NORMAL_PAN::get_parameter(int param) { 
  switch (param) {
  case 1: 
    return(r_kerroin * 100.0);
  }
}

void EFFECT_NORMAL_PAN::process(SAMPLE_BUFFER::sample_type *insample) {
  insample->put_left(insample->get_left() * l_kerroin);
  insample->put_right(insample->get_right() * r_kerroin);
}

EFFECT_NORMALIZE::EFFECT_NORMALIZE (void) {
    max = 0;
    kerroin = 1.0;

    ecadebug->msg("(fx) Normalizing; analyzing sample data.");
}

void EFFECT_NORMALIZE::status(string* output) {
  if (max != 0.0) kerroin = SAMPLE_BUFFER::max_amplitude / max;
  if (kerroin < 1.0) kerroin = 1.0;
  MESSAGE_ITEM otemp;
  otemp.setprecision(0);
  otemp << "(fx) Max amplitude " << max;
  otemp << "; signal can be amplified by " << kerroin * 100.0;
  otemp << "%.";
  *output = *output + otemp.to_string();
}

void EFFECT_NORMALIZE::process(SAMPLE_BUFFER::sample_type *insample) {
  if (fabs(insample->get_left()) > max) max = fabs(insample->get_left());
  if (fabs(insample->get_right()) > max) max = fabs(insample->get_right());    
}
