// ------------------------------------------------------------------------
// ecatools_convert.cpp: A simple command-line tool for converting
//                       audio files.
// Copyright (C) 2000 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <string>
#include <cstdio>

#include <kvutils/com_line.h>

#include <eca-debug.h>
#include <eca-error.h>
#include <eca-control.h>
#include <eca-engine.h>
#include <eca-session.h>
#include <eca-version.h>

#include "ecatools_convert.h"

static const string ecatools_play_version = "20011009";

int main(int argc, char *argv[])
{
#ifdef NDEBUG
  ecadebug->disable();
#else
  ecadebug->set_debug_level(ECA_DEBUG::info |
			    ECA_DEBUG::module_flow);
#endif

  COMMAND_LINE cline = COMMAND_LINE (argc, argv);

  if (cline.size() < 2) {
    print_usage();
    return(1);
  }

  try {
    string filename;

    ECA_SESSION esession;
    ECA_CONTROL ectrl (&esession);
    ECA_AUDIO_FORMAT aio_params;

    cline.begin();
    cline.next(); // skip the program name

    string extension (".raw");
    if (cline.end() != true) {
      extension = cline.current();
      cline.next();
    }

    while(cline.end() != true) {
      filename = cline.current();
      std::cerr << "Converting file \"" << filename << "\" --> ";
      std::cerr << "\"" << filename + extension << "\"." << std::endl;

      ectrl.add_chainsetup("default");
      ectrl.add_chain("default");
      ectrl.add_audio_input(filename);
      if (ectrl.get_audio_input() == 0) {
	std::cerr << "---\nError while processing file " << filename << ". Exiting...\n";
	break;
      }
      ectrl.set_default_audio_format_to_selected_input();
      aio_params = ectrl.default_audio_format();
      ectrl.set_chainsetup_parameter("-sr:" + kvu_numtostr(aio_params.samples_per_second()));
      ectrl.add_audio_output(filename + extension);
      if (ectrl.get_audio_output() == 0) {
	std::cerr << "---\nError while processing file " << filename + extension << ". Exiting...\n";
	break;
      }
      ectrl.connect_chainsetup();
      if (ectrl.is_connected() == false) {
	std::cerr << "---\nError while converting file " << filename << ". Exiting...\n";
	break;
      }

      ECA_ENGINE emain (&esession);      
      emain.exec();

      ectrl.disconnect_chainsetup();
      ectrl.remove_chainsetup();

      cline.next();
    }
  }
  catch(ECA_ERROR& e) {
    std::cerr << "---\nERROR: [" << e.error_section() << "] : \"" << e.error_message() << "\"\n\n";
  }
  catch(...) {
    std::cerr << "\nCaught an unknown exception.\n";
  }
  return(0);
}

void print_usage(void) {
  std::cerr << "****************************************************************************\n";
  std::cerr << "* ecatools_convert, v" << ecatools_play_version;
  std::cerr << " (linked to ecasound v" << ecasound_library_version 
       << ")\n";
  std::cerr << "* (C) 1997-2001 Kai Vehmanen, released under GPL licence \n";
  std::cerr << "****************************************************************************\n";

  std::cerr << "\nUSAGE: ecaconvert .extension file1 [ file2, ... fileN ]\n\n";
}
